/*/////////////////////////////////////////////////////////////////////////////
//! \file fcBase.h:
//! \brief Header File for the fcBase API.
//!
//! Use this header file for developing user defined FlexCard applications.
//! See also printed documentation for more information. The common data types
//! and enumerations can be found in the header file "fcBaseTypes.h".
//!
//! Generally:
//!
//! - Each function of this library (except some error handling functions)
//!   returns an error code. If the return value is equal to 0, no error
//!   occured. A number greater than 0 indicates an error. To get more
//!   information about it, use the error handling functions.
//!
//! - Some functions will allocate memory for you. In such a case the
//!   fcFreeMemory function need to be called to release this memory.
//!
//! \par Overview:
//! Author:         EBEL/WO/SH/Ba/TB/Tr/LJ
//! $Date: 2009-02-16 13:30:23 +0100 (Mo, 16 Feb 2009) $ <br>
//! \par Copyright:
//! Copyright 2008-2009 Eberspaecher Electronics GmbH & Co. KG
//! \version $Revision: 15010 $
/////////////////////////////////////////////////////////////////////////////*/

#ifndef _FCBASE_H_
#define _FCBASE_H_

#ifndef _FCBASE_TYPES_H_
    #error Please include first the header file "fcBaseTypes.h"
#endif 


// Linux
#ifdef linux

	#ifndef FCBASE_API
		#define FCBASE_API
	#endif
	//! Please do not use the FCBASE_EXPORTS define in your application, unless
	//! you have the source code ;-)
	#ifdef FCBASE_EXPORTS
		#define EXPORTDLL extern "C" __attribute__ ((visibility ("default")))
	#else
	    #define EXPORTDLL extern "C" 
	#endif

// Windows
#else

	#ifndef FCBASE_API
		#define FCBASE_API __stdcall
	#endif
	//! Please do not use the FCBASE_EXPORTS define in your application, unless
	//! you have the source code ;-)
	#ifdef FCBASE_EXPORTS
		#define EXPORTDLL       extern "C" __declspec(dllexport)
	#else
		#define EXPORTDLL       extern "C" __declspec(dllimport)
	#endif

#endif

//----------------------------------------------------------------------
//          Error handling
//----------------------------------------------------------------------

//! This function returns the error type for a given error. Please, refer to
//! #fcErrorType for more details. 
//! \param error [IN] <br>  An error value of type fcError
//! \return Type of error
//! \see #fcErrorType, #fcGetErrorCode, #fcGetErrorText
EXPORTDLL fcErrorType FCBASE_API fcGetErrorType(fcError error);

//! This function returns the error code for a given error. A zero value
//! indicates no error occured. The list of all error codes can be found in the
//! #fcErrorCode enumeration (see fcBaseTypes.h).
//! \param error [IN] <br> An error value of type fcError
//! \return Error code
//! \see #fcErrorCode, #fcGetErrorType, #fcGetErrorText
EXPORTDLL fcErrorCode FCBASE_API fcGetErrorCode(fcError error);

//! This function returns the corresponding error text for a given error. To
//! free the memory which was allocated by this function, please use the
//! function #fcFreeMemory with the type #fcMemoryTypeString (see fcMemoryType).
//! Some text will be generated at runtime to provide a more detailed error
//! description
//! \param error [IN] <br> An error value of type fcError for which an error
//! 	   text should be returned.
//! \param szText [OUT] <br> Address of char pointer which holds the adress for
//!        the constructed error text.
//! \return If the function succeeds (return value is zero), the parameter
//!         szText contains the requested error text. If the function fails
//!         szText is not valid. The fcErrorCode NULL_PARAMETER is returned if
//!         the szText parameter is a null pointer, TEXT_NOT_DEFINIED if no
//!         error text for the given error could be found, or
//!         MEMORY_ALLOCATION_FAILED to indicate that the memory allocation for
//!         the error text failed.
//! \see #fcFreeMemory, #fcGetErrorType, #fcGetErrorCode
EXPORTDLL fcError FCBASE_API fcGetErrorText(fcError error,
											char** szText);

//----------------------------------------------------------------------
//          Memory handling
//----------------------------------------------------------------------

//! This function releases the memory allocated by one of the API functions
//! #fcGetErrorText, #fcbGetEnumFlexCards, #fcbGetEnumFlexCardsV2,
//! #fcbGetEnumFlexCardsV3, #fcbGetInfoFlexCard or #fcbReceive. The allocated
//! memory can be used as long as necessary. If the connection to the FlexCard
//! is closed, all allocated memory blocks must be released with this function.
//! \param memoryType Type of memory to be released.
//! \param p Pointer to the memory to be released.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information. The fcErrorCode INVALID_PARAMETER is
//!         returned if the memoryType parameter was not correct, NULL_PARAMETER
//!         if the p parameter is a null pointer.
//! \see #fcMemoryType, #fcGetErrorText, #fcbGetEnumFlexCards, #fcbGetEnumFlexCardsV2,
//!      #fcbGetEnumFlexCardsV3, #fcbGetInfoFlexCard, #fcbReceive
EXPORTDLL fcError FCBASE_API fcFreeMemory(const fcMemoryType memoryType,
                                          void* p);

//----------------------------------------------------------------------
//           Initialization 
//----------------------------------------------------------------------

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbGetEnumFlexCardsV3 now.)
//! This function returns a linked list of the unused FlexCards found on the
//! system. To free the memory which was allocated by this function, please use
//! the function #fcFreeMemory with type #fcMemoryTypeInfo.
//! \param pInfo [OUT] <br> linked list of #fcInfo objects
//! \return If the function succeeds, the return value is 0. If the function
//!         fails the content of pInfo is not valid. The error code
//!         NULL_PARAMETER is returned if the pInfo parameter is a null pointer.
//!         If the memory allocation in this function fails the error code
//!         MEMORY_ALLOCATION_FAILED is returned. 
//! \remarks If a connection to a FlexCard is already opened, this FlexCard does
//!          not appear in this list. If the function succeeds, there will be
//! 		 always one valid fcInfo structure regardless if there is a FlexCard
//! 		 in the system or not. This functionality is given to provide
//!          version information about the DLL. The version information
//!          concerning the hardware is only valid if the identifier
//!          (pInfo->FlexCardId) is higher than 0.
//! \note This function allocates memory for use. To prevent memory leaks you
//!       have to free it up by calling fcFreeMemory with the type
//! 	  fcMemoryTypeInfo.
//! \note Internally, the function uses the #fcbGetEnumFlexCardsV3 function.
EXPORTDLL fcError FCBASE_API fcbGetEnumFlexCards(fcInfo** pInfo);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbGetEnumFlexCardsV3 now.)
//! This function returns a linked list of the installed FlexCards found on the
//! system. To free the memory which was allocated by this function, please use
//! the function #fcFreeMemory with type #fcMemoryTypeInfoV2.
//! \param pInfoV2 [OUT] <br> linked list of #fcInfoV2 objects
//! \return If the function succeeds, the return value is 0. If the function
//!         fails the content of pInfoV2 is not valid. The error code
//!         NULL_PARAMETER is returned if the pInfoV2 parameter is a null pointer.
//!         If the memory allocation in this function fails the error code
//!         MEMORY_ALLOCATION_FAILED is returned.
//! \remarks If the function succeeds, there will be always one valid fcInfoV2
//!          structure regardless if there is a FlexCard in the system or not.
//!          This functionality is given to provide version information about
//!          the DLL / library. The version information concerning the hardware
//!          is only valid if the identifier (pInfoV2->FlexCardId) is higher than 0.
//! \note This function allocates memory for use. To prevent memory leaks you
//!       have to free it up by calling fcFreeMemory with the type fcMemoryTypeInfoV2.
//! \note Internally, the function uses the #fcbGetEnumFlexCardsV3 function.
EXPORTDLL fcError FCBASE_API fcbGetEnumFlexCardsV2(fcInfoV2** pInfoV2);

//! This function returns a linked list of the FlexCards found on the system.
//! To free the memory which was allocated by this function, please use the
//! function #fcFreeMemory with type #fcMemoryTypeInfoHwSw.
//! \param pInfoHwSw [OUT] <br> linked list of #fcInfoHwSw objects
//! \param getBusyDevices [IN] <br> Show busy devices in linked list. Set this parameter
//!        to false to get a linked list of the unused FlexCards found on the system.
//! \return If the function succeeds, the return value is 0. If the function
//!         fails the content of pInfoHwSw is not valid. The error code
//!         NULL_PARAMETER is returned if the pInfoHwSw parameter is a null pointer.
//!         If the memory allocation in this function fails the error code
//!         MEMORY_ALLOCATION_FAILED is returned.
//! \remarks If the function succeeds, there will be always one valid fcInfoHwSw
//!          structure regardless if there is a FlexCard in the system or not.
//!          This functionality is given to provide version information about
//!          the DLL / library. The version information concerning the hardware
//!          is only valid if the identifier (pInfoHwSw->FlexCardId) is higher than 0.
//! \note This function allocates memory for use. To prevent memory leaks you
//!       have to free it up by calling fcFreeMemory with the type fcMemoryTypeInfoHwSw.
EXPORTDLL fcError FCBASE_API fcbGetEnumFlexCardsV3(fcInfoHwSw** pInfoHwSw,
												   bool getBusyDevices);

//! This function checks the version of dll, sys and firmware.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
EXPORTDLL fcError FCBASE_API fcbCheckVersion(fcHandle hFlexCard);

//! This function opens a connection to a specific FlexCard and returns a handle
//! to his FlexCard. The function modifies some communication controller
//! registers (e.g. set the communication controller in its configuration state,
//! fcStateConfig) and all message buffer are configured as receive FIFO buffers
//! with maximum payload length.
//! \param phFlexCard [OUT] <br> Handle to a specific FlexCard.
//! \param flexCardId [IN] <br> Number which indicates the FlexCard you want to
//!        use. This identifier is stored in #fcInfoHwSw objects returned by the
//!        function #fcbGetEnumFlexCardsV3. Only FlexCardId greater than zero are
//!        valid FlexCard identifier.
//! \return If the function succeeds, phFlexCard holds a valid FlexCard handle
//!         and the return value is 0. If the value <> 0, please use the
//!         functions described in the section Error Handling to get extended
//!         error information.
//! \remarks Use the function #fcbGetEnumFlexCardsV3 to get a valid FlexCardId.
//!          The function #fcbClose is used to close a connection previously
//!          opened with #fcbOpen
//! \note If the FlexCard is closed and reopened all previous (before closing)
//!       configuration settings are lost. After opening a connection it is
//! 	  necessary to configure the FlexCard.
//! \see #fcbGetEnumFlexCardsV3, #fcbClose
EXPORTDLL fcError FCBASE_API fcbOpen(fcHandle* phFlexCard, 
                                     fcDword flexCardId);

//! This function closes the connection to a FlexCard.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         please use the functions described in the section Error Handling to
//!         get extended error information.
//! \remarks If a monitoring is active, this function will first stop the
//!          monitoring and then close the connection.
//! \see #fcbGetEnumFlexCardsV3, #fcbOpen
EXPORTDLL fcError FCBASE_API fcbClose(fcHandle hFlexCard);

//! This function returns an item of the structure #fcInfoHwSw, which provides
//! information about the components, the identifiers and the current device state
//! of the opened FlexCard device. To free the memory which was allocated by this
//! function, please use the function #fcFreeMemory with type #fcMemoryTypeInfoHwSw.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param pInfoHwSw [OUT] <br> Hard- and software information of a FlexCard.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         please use the functions described in the section Error Handling to
//!         get extended error information.
//! \see #fcbGetEnumFlexCardsV3, #fcbOpen
//! \note This function allocates memory for use. To prevent memory leaks you
//!       have to free it up by calling fcFreeMemory with the type fcMemoryTypeInfoHwSw.
EXPORTDLL fcError FCBASE_API fcbGetInfoFlexCard(fcHandle hFlexCard, fcInfoHwSw** pInfoHwSw);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRMonitoringStart now.)
//! This function is used to start the monitoring of a FlexRay bus. Once called,
//! the function changes the communication controller state from configuration
//! state to normal active state (if the cluster integration succeeds). The
//! current communication controller state can be read using the function
//! #fcbGetCcState. If the FlexCard is synchronized with the cluster the
//! function #fcbGetCcState will return the value #fcStateNormalActive.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param mode [IN] <br> The monitoring mode. Not every monitoring mode is
//!        supported by all communication controllers. See #fcMonitoringModes
//! 	   for details
//! \param restartTimestamps [IN] <br> Set this parameter to true to restart the
//! 	   measurement without resetting the FlexCard timestamp. Set it to false
//! 	   to start the measurement from the beginning. The timestamps have
//! 	   micro second resolution.
//! \param enableCycleStartEvents [IN] <br> Set this parameter to true to enable
//! 	   the cycle start events in order that at the beginning of every cycle
//! 	   the event fcNotificationTypeFRCycleStarted is signalled.
//! \param enableColdstart [IN] <br> Set this parameter to true to allow the
//! 	   FlexCard to initialize the cluster communication, otherwise the
//! 	   coldstart inhibit mode is active. This feature can not be used in the
//! 	   monitoring mode fcMonitoringDebug and fcMonitoringDebugAsynchron.
//! \param enableWakeup [IN] <br> Set this parameter to true to transmit a
//! 	   wakeup pattern on the configured wakeup channel (FlexRay Protocol
//! 	   Specification: pWakeupChannel). A cluster wakeup must precede the
//! 	   communication start up to ensure that all nodes in a cluster are
//! 	   awake. The minimum requirement for a cluster wakeup is that all bus
//! 	   drivers are supplied with power. This feature can not be used in the
//! 	   monitoring mode fcMonitoringDebug and fcMonitoringDebugAsynchron.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         please use the functions described in the section Error Handling to
//!         get extended error information.
//! \remarks After the monitoring has started, the user should check if the
//!          integration in the cluster was successful: #fcbGetCcState should
//!          return the state #fcStateNormalActive. 
//! \note After the monitoring has successfully started, the receive process has
//! 	  to be started as soon as possible to avoid an overflow (error packet
//! 	  fcErrFlexcardOverflow is received). Once an overflow occurred, no more
//! 	  packets can be received. The monitoring has to be stopped and started
//! 	  again.
//! \see #fcbMonitoringStop, #fcbGetCcState, #fcMonitoringModes,
//!      #fcbSetEventHandle
//! \note Internally, the function uses the #fcbFRMonitoringStart function.
EXPORTDLL fcError FCBASE_API fcbMonitoringStart(fcHandle hFlexCard,
                                                fcMonitoringModes mode, 
                                                bool restartTimestamps,
                                                bool enableCycleStartEvents,
                                                bool enableColdstart,
                                                bool enableWakeup);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRMonitoringStop now.)
//! This function stops the FlexRay bus measurement. The communication
//! controller is set back in its configuration state.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
//! \see #fcbMonitoringStart
//! \note Internally, the function uses the #fcbFRMonitoringStop function.
EXPORTDLL fcError FCBASE_API fcbMonitoringStop(fcHandle hFlexCard);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRGetCcState now.)
//! This function returns the current communication controller POC state. For a
//! description of possible state, refer to the enumeration #fcState. This
//! function should be used to check if the integration into a FlexRay cluster
//! has succeeded.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param pState [OUT] <br> Current communication controller state
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         please use the functions described in the section Error Handling to
//!         get extended error information.
//! \see #fcbMonitoringStart, #fcbMonitoringStop
//! \note Internally, the function uses the #fcbFRGetCcState function.
EXPORTDLL fcError FCBASE_API fcbGetCcState(fcHandle hFlexCard,
										   fcState* pState);

//! This function returns the current time stamp of the FlexCard device and the
//! correlated performance counter value of the operating system.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param pTimeStamp [OUT] <br> Current time stamp
//!		   pPerformanceCounter [OUT] <br> Correlated performance counter
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         please use the functions described in the section Error Handling to
//!         get extended error information.
EXPORTDLL fcError FCBASE_API fcbGetCurrentTimeStamp(fcHandle hFlexCard,
													fcDword* pTimeStamp,
													fcQuad* pPerformanceCounter);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRSetTransceiverState now.)
//! Sets the transceiver mode individually for each channel.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param stateChannelA [IN] <br> The new transceiver state for channel A.
//! \param stateChannelB [IN] <br> The new transceiver state for channel B.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         please use the functions described in the section Error Handling to
//!         get extended error information. 
//! \remarks If one of the transceiver is in the sleep mode and the transceiver
//! 		 detects a wakeup event, the notification event
//! 		 fcNotificationTypeFRWakeup is fired once only.
//! \note Internally, the function uses the #fcbFRSetTransceiverState function.
EXPORTDLL fcError FCBASE_API fcbSetTransceiverState(fcHandle hFlexCard, 
													fcTransceiverState stateChannelA, 
													fcTransceiverState stateChannelB);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRGetTransceiverState now.)
//! Gets the current transceiver state for each channel.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param stateChannelA [OUT] <br> The current transceiver state for channel A.
//! \param stateChannelB [OUT] <br> The current transceiver state for channel B.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         please use the functions described in the section Error Handling to
//!         get extended error information. 
//! \remarks If one of the transceiver is in the sleep mode and the transceiver
//! 		 detects a wakeup event, the notification event
//! 		 fcNotificationTypeFRWakeup is fired once only.
//! \note Internally, the function uses the #fcbFRGetTransceiverState function.
EXPORTDLL fcError FCBASE_API fcbGetTransceiverState(fcHandle hFlexCard,	
													fcTransceiverState* stateChannelA, 
													fcTransceiverState* stateChannelB);



//----------------------------------------------------------------------
//          Configuration
//----------------------------------------------------------------------

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRSetCcRegister now.)
//! This function writes a value in a given register of the communication
//! controller. Not every register can be written (e.g. the registers belonging
//! to the message buffer configuration or some interrupt settings). 
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param address [IN] <br> Address of the CC register to be written. 
//! \param value [IN] <br> The value to be written.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         please use the functions described in the section Error Handling to
//!         get extended error information. If the register can not be written
//!         the error code REGISTER_NOT_WRITEABLE is returned
//! \remarks For a register description refer to the specification of the
//!          corresponding communication controller. Modifying one of the
//!          following registers will reset message buffers with their default
//!          settings (FIFO receive buffers). The users message buffers.
//!          configuration will not be valid anymore: 
//!          - Bosch E-Ray: MHDC (0x0098) and GTUC7 (0x00B8). 
//!          - Freescale FPGA: Freescale FPGA: SPLR (0x090), NSSR (0x08E), and
//!            MPLDR (0x092). 
//! \note: Not all register of a communication controller can be
//!        set. The base API will modify some parameters so that the operating
//!        of the FlexCard is guaranteed (e.g. interrupt settings). Access is
//!        denied to all register which are used for message buffer
//!        configuration.
//! \see #fcbGetCcRegister
//! \note Internally, the function uses the #fcbFRSetCcRegister function.
EXPORTDLL fcError FCBASE_API fcbSetCcRegister(fcHandle hFlexCard, 
                                              fcDword address,
                                              fcDword value);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRGetCcRegister now.)
//! This function reads and returns the content of a given register of the
//! communication controller.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param address [IN] <br> Address of the CC register to be read. 
//! \param pValue [OUT] <br> The content of the desired CC register.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information. If the register can not be read the
//!         error code REGISTER_NOT_READABLE is returned.
//! \remarks Not every register could be read. For a register description refer
//!          to the specification of the corresponding communication controller.
//! \note Internally, the function uses the #fcbFRGetCcRegister function.
EXPORTDLL fcError FCBASE_API fcbGetCcRegister(fcHandle hFlexCard, 
                                              fcDword  address,
                                              fcDword* pValue);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRConfigureMessageBuffer now.)
//! This function configures transmit, receive and FIFO message buffers of the
//! communication controller. Configuring message buffers is only allowed if the
//! communication controller is in its configuration state, fcStateConfig.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param bufferId <br> [OUT] Message buffer identifier. If the configuration
//! 	   was successful the message buffer identifier is greater than 0. This
//! 	   identifier will be required to transmit the content of the buffer (in
//! 	   the case of a transmit buffer).
//! \param cfg [IN] <br> Message buffer configuration parameters.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \remarks Before configuring the message buffers, it is necessary to set up
//! 		 the global communication parameters (cluster parameters).
//! 		 Internally the FlexCard used the FIFO buffers as receive buffers,
//! 		 therefore we recommend using FIFO message buffers as much as
//! 		 possible.
//! \note Internally, the function uses the #fcbFRConfigureMessageBuffer function.
EXPORTDLL fcError FCBASE_API fcbConfigureMessageBuffer(fcHandle hFlexCard, 
                                                       fcDword* bufferId,
                                                       fcMsgBufCfg cfg);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRReconfigureMessageBuffer now.)
//! This function reconfigures transmit, receive and FIFO message buffers of the
//! communication controller. A reconfiguration is only allowed for message
//! buffers which are already configured.. This function is available in all
//! states of the CC. Not all configuration settings could be modified in
//! monitoring state. Refer to the documentation of the message buffer
//! structures for further details.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param bufferId <br> [IN] The unique id of the message buffer which should
//!        be reconfigured. 
//! \param cfg [IN] <br> Message buffer configuration parameters.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \note Internally, the function uses the #fcbFRReconfigureMessageBuffer function.
EXPORTDLL fcError FCBASE_API fcbReconfigureMessageBuffer(fcHandle hFlexCard, 
                                                         fcDword bufferId,
                                                         fcMsgBufCfg cfg);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRGetMessageBuffer now.)
//! This function reads a specific message buffer configuration.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param bufferId [IN] <br> The identifier of the message buffer to be read.
//! \param cfg [OUT] <br> The configuration parameters of the specific message
//!        buffer.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \note Internally, the function uses the #fcbFRGetMessageBuffer function.
EXPORTDLL fcError FCBASE_API fcbGetCcMessageBuffer(fcHandle hFlexCard, 
                                                   fcDword bufferId, 
                                                   fcMsgBufCfg* cfg);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRResetMessageBuffers now.)
//! This function resets the communication controller message buffers. After
//! calling this function, all message buffers are configured as receive FIFO -
//! with maximal payload (depends on the communication controller).
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \note Internally, the function uses the #fcbFRResetMessageBuffers function.
EXPORTDLL fcError FCBASE_API fcbResetCcMessageBuffer(fcHandle hFlexCard);

//! This function re-initializes the message buffer configuration of the specified
//! bus type and communication controller index. After calling this function the
//! communication controller does not send old payload data. Re-initialization of message
//! buffers is only allowed if the communication controller is in configuration
//! state.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param BusType [IN] <br> The bus type.
//! \param CC [IN] <br> Index of the communication controller.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
EXPORTDLL fcError FCBASE_API fcbReinitializeCcMessageBuffer(fcHandle hFlexCard,
															fcBusType BusType,
															fcCC CC);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRSetCcConfigurationCANdb now.)
//! This function configures the communication controller of the FlexCard with a
//! CANdb compatible string. Before the configuration of the communication
//! controller starts, all message buffers are reset to their default settings
//! (FIFO buffer).
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param szCanDb [IN] <br> Pointer to null-terminated CANdb string
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \note Internally, the function uses the #fcbFRSetCcConfigurationCANdb function.
EXPORTDLL fcError FCBASE_API fcbCanDbCcConfiguration(fcHandle hFlexCard, 
                                                     const char* szCanDb);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRSetCcConfigurationChi now.)
//! This function configures the communication controller of the FlexCard with a
//! FlexConfig compatible configuration string (CHI File). The payload data for
//! transmit message buffers are not set by this function. Before the
//! configuration of the communication controller starts, all message buffers
//! are reset to their default settings (FIFO buffer).
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param szChi [IN] <br> Pointer to null-terminated CHI string (refer to the
//! 	   CHI string example section).
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \note Internally, the function uses the #fcbFRSetCcConfigurationChi function.
EXPORTDLL fcError FCBASE_API fcbChiCcConfiguration(fcHandle hFlexCard, 
                                                   const char* szChi);

//! THIS FUNCTION IS OBSOLETE  (Please use fcTriggerConditionEx, fcTriggerConfigurationEx and fcbSetTrigger() now.)
//! This function configures and starts/stops a trigger. For further
//! information, refer to the structures #fcTriggerCfgSoftware and
//! #fcTriggerCfgHardware.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param enable [IN] <br> Set to true to enable the trigger, and to false to
//! 	   disable it.
//! \param cfg [IN] <br> The trigger configuration.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
EXPORTDLL fcError FCBASE_API fcbTrigger(fcHandle hFlexCard,
                                        bool enable,
                                        fcTriggerCfg cfg);

//! This function configures and starts/stops a trigger. For further
//! information, refer to the structure #fcTriggerConfigurationEx
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param cfg [IN] <br> The trigger configuration.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
EXPORTDLL fcError FCBASE_API fcbSetTrigger(fcHandle hFlexCard,
                                           fcTriggerConfigurationEx cfg);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRSetSoftwareAcceptanceFilter now.)
//! This function configures the frame ids accepted by the device driver. Only
//! the ids which are in the filter list are forwarded to the user application,
//! all other frames are rejected. To accept all frames set the parameters pData
//! to NULL and size to zero or configure a single frame id of zero.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param channel [IN] <br> FlexCard channel(s) concerned by the filter.
//! \param pData [IN] <br> Pointer to an fcDword array containing the ids which
//!        are accepted. Each element (fcDword) contains one frame identifier.
//!        --------------------------------            
//!        |  fcDword  |  fcDword  | ...                
//!        --------------------------------                
//!        |   ID x    |   ID y    | ...                
//!        --------------------------------                
//! \param size [IN] <br> Number of ids in the array 
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
//! \note Internally, the function uses the #fcbFRSetSoftwareAcceptanceFilter function.
EXPORTDLL fcError FCBASE_API fcbFilter(fcHandle hFlexCard,
									   fcChannel channel,
									   fcDword* pData,
									   fcDword size);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRSetCcTimerConfig now.)
//! This function configures the communication controller timer interrupt. To
//! get a notification when the cc timer interval has occurred, an event of type
//! #fcNotificationTypeFRCcTimer has to be registered by the function
//! #fcbSetEventHandle. The cc timer can be enabled/disabled by this function.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param cfg [IN] <br> The cc timer configuration.
//! \param bEnable [IN] <br> Set to true to enable the cc timer, and to false to
//! 	   disable it.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
//! \see #fcbSetEventHandle #fcCcTimerCfg #fcbGetCcTimerConfig
//! \note Internally, the function uses the #fcbFRSetCcTimerConfig function.
EXPORTDLL fcError FCBASE_API fcbSetCcTimerConfig(fcHandle hFlexCard,
												 fcCcTimerCfg cfg,
												 bool bEnable);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRGetCcTimerConfig now.)
//! This function reads the communication controller timer configuration.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param pCfg [OUT] <br> The configuration parameters of the cc timer.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \see #fcCcTimerCfg, #fcbSetCcTimerConfig
//! \note Internally, the function uses the #fcbFRGetCcTimerConfig function.
EXPORTDLL fcError FCBASE_API fcbGetCcTimerConfig(fcHandle hFlexCard,
												 fcCcTimerCfg* pCfg);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRCalculateMacrotickOffset now.)
//! This function calculates the macrotick offset for a specific cycle
//! position in a FlexRay cycle.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param CyclePosition [IN] <br> The cycle position.
//! \param SlotOrMiniSlotId [IN] <br> This parameter is used for a cycle
//!	   position of fcCyclePosStaticSlot and fcCyclePosDynamicMiniSlot
//!	   to calculate the macrotick offset for a static slot or a dynamic
//!	   mini slot id.
//! \param pValue [OUT] <br> The macrotick offset value.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \see #fcCyclePos, #fcCcTimerCfg, #fcbSetCcTimerConfig
//! \note Internally, the function uses the #fcbFRCalculateMacrotickOffset function.
EXPORTDLL fcError FCBASE_API fcbCalculateMacrotickOffset(fcHandle hFlexCard,
														 fcCyclePos CyclePosition,
														 fcDword SlotOrMiniSlotId,
														 fcDword* pValue);

//! This function configures the packet overflow handling of the FlexCard. The FlexCards
//! default behavior is to stop the monitoring if a buffer overflow was detected. This
//! function can configure the FlexCard to continue with the monitoring when an amount
//! of free RAM space is available again. An error packet fcErrFlexCardOverflow is
//! generated in both cases.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param bContinue [IN] <br> Set this flag to true to continue the monitoring in case
//!		   of a packet buffer overflow is detected when RAM space is available again.
//!		   Set to false to stop the monitoring.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
EXPORTDLL fcError FCBASE_API fcbSetContinueOnPacketOverflow(fcHandle hFlexCard,
															bool bContinue);

//! This function reads the number of the various communication controllers which are 
//! available on the FlexCard.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param pNumberCC [OUT] <br> Pointer to the structure of the available communication
//!		   controller numbers.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
EXPORTDLL fcError FCBASE_API fcbGetNumberCcs(fcHandle hFlexCard,
											 fcNumberCC* pNumberCC);

//! This function resets the timestamp on the FlexCard.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
EXPORTDLL fcError FCBASE_API fcbResetTimestamp(fcHandle hFlexCard);

//----------------------------------------------------------------------
//          Events
//----------------------------------------------------------------------

//! This function registers an event handle for a specific notification type.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param hEvent [IN] <br> Event handle to be registered to signal when a new
//! 	   cycle starts or a timer interval has elapsed depending on the given
//! 	   type.
//! \param type [IN] <br> The notification type for which the event has to be
//! 	   registered.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
//! \see #fcNotificationType
EXPORTDLL fcError FCBASE_API fcbSetEventHandle(fcHandle hFlexCard, 
                                               fcHandle hEvent,
                                               fcNotificationType type);

//! This function registers an event handle for a specific notification type.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param CC [IN] <br> Communication controller identifier.
//! \param hEvent [IN] <br> Event handle to be registered to signal when a new
//! 	   cycle starts or a timer interval has elapsed depending on the given
//! 	   type.
//! \param type [IN] <br> The notification type for which the event has to be
//! 	   registered.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
//! \see #fcNotificationType
EXPORTDLL fcError FCBASE_API fcbSetEventHandleV2(fcHandle hFlexCard,
												 fcCC CC,
												 fcHandle hEvent,
												 fcNotificationType type);

//! This function enables or disables the internal FlexCard timer. To become
//! notify when timer interval has elapsed, an event of type
//! #fcNotificationTypeTimer has to be register by the function
//! #fcbSetEventHandle.
//! \param hFlexCard  [IN] <br> Handle to a FlexCard
//! \param enable  [IN] <br> Set to true to enable the timer and to false to disable it.
//! \param timerInterval [IN] <br> Specifies the timer period in us
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
//! \note This feature is not supported by the FlexCard Cyclone I Card. 
//! \see #fcbSetEventHandle
EXPORTDLL fcError FCBASE_API fcbSetTimer(fcHandle hFlexCard, 
                                         bool enable,
                                         fcDword timerInterval);


//! This function generates a notification packet each time the configured timer
//! timeout has elapsed. This timer can be enabled/disabled by this function and
//! the timeout can be set. The notification packets are inserted in the stream
//! and received through the function #fcbReceive.
//! \param hFlexCard  [IN] <br> Handle to a FlexCard
//! \param enable [IN] <br> Set to true to enable the timer and to false to
//! 	   disable it.
//! \param timerInterval [IN] Specifies the time-out interval, in microseconds.
//! 	   A packet is generated as soon as the time-out has elapsed. The timer
//! 	   interval must be greater than 50us and smaller than 655350us. The value
//! 	   must be rounded to 10us units.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
EXPORTDLL fcError FCBASE_API fcbNotificationPacket(fcHandle hFlexCard, 
												   bool enable,
												   fcDword timerInterval);



//----------------------------------------------------------------------
//          Receive / Transmit
//----------------------------------------------------------------------

//! This function reads all available packets from the FlexCard memory into a
//! memory block allocated by the fcBase API. The frames are stored into a
//! linked list. To free the memory allocated by this function, use the function
//! #fcFreeMemory with the type #fcMemoryTypePacket.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param pPacket [OUT] <br> Address of the fcPacket object pointer. The memory
//!        for this structure and its content is allocated by the fcBase API.
//!        Packets are available if the return code is 0 and pPacket is not a
//!        null pointer.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
//! \note This function allocates memory for use. To prevent memory leaks you
//!       have to free it up by calling fcFreeMemory with the type
//! 	  fcMemoryTypePacket.
EXPORTDLL fcError FCBASE_API fcbReceive(fcHandle hFlexCard,
										fcPacket** pPacket);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRTransmit now.)
//! This function writes a data frame into a communication controller transmit
//! buffer of the FlexCard. The frame should normally be transmitted in the next
//! cycle. If the transmit acknowledgment is activated, an acknowledge packet is
//! generated as soon as the frame has been transmitted. This function should
//! only be called when the FlexCard is in normal active state or when all
//! message buffer configuration has been done.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param bufferId [IN] <br> The id of the message buffer used for the
//!        transmission.
//! \param payload [IN] The payload data to be transmitted.
//! \param payloadLength [IN] <br> The size of the payload data (number of
//!        2-byte words)
//! \remarks The payload data has to be organized as follows: if Data0 is the
//!          first byte to transmit and Data1 the second byte to transmit, then
//!          the high byte (Bit 8 - 15) of payload[0] contains Data1, the low
//!          byte (Bit0-7) of payload[0] contains Data0, etc.
//!          -----------------------------            
//!          |   Word 0  |   Word 1  | ...                
//!          -----------------------------                
//!          |Data1|Data0|Data3|Data2| ...                
//!          -----------------------------                
//!
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
//! \note Internally, the function uses the #fcbFRTransmit function.
EXPORTDLL fcError FCBASE_API fcbTransmit(fcHandle hFlexCard,
                                         fcDword bufferId,
                                         fcWord payload[],
                                         fcByte payloadLength);

//! THIS FUNCTION IS OBSOLETE  (Please use #fcbFRTransmitSymbol now.)
//! This function transmits a symbol in the symbol window segment. It can only
//! be called if the communication controller is in the POC state NORMAL_ACTIVE.
//! For a list of available symbols to be transmitted, see the enumeration
//! #fcSymbolType.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param symbol [IN] <br> Type of symbol to transmit
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
//! \note Internally, the function uses the #fcbFRTransmitSymbol function.
EXPORTDLL fcError FCBASE_API fcbTransmitSymbol(fcHandle hFlexCard,
                                               fcSymbolType symbol);

//! This function configures additional startup/sync transmit message buffers 
//! for self synchronization. Configuring message buffers is only allowed if the
//! communication controller is in its configuration state, fcStateConfig.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param bufferId <br> [OUT] Message buffer identifier. If the configuration
//! 	   was successful the message buffer identifier is greater than 0. This
//! 	   identifier will be required to transmit the content of the buffer (in
//! 	   the case of a transmit buffer).
//! \param cfg [IN] <br> Message buffer configuration parameters.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \remarks Before configuring the message buffers, it is necessary to set up
//! 		 the global communication parameters (cluster parameters).
//! 		 possible. Loading a new chi-File will reset the additional
//!          startup/sync message buffer configurations.
EXPORTDLL fcError FCBASE_API fcbConfigureMessageBufferSelfSynchronization(fcHandle hFlexCard, 
																		  fcDword* bufferId,
																		  fcMsgBufCfg cfg);

//! This function reconfigures the additional startup/sync transmit buffers.
//! A reconfiguration is only allowed for message buffers which are already 
//! configured and if the communication controller is in its configuration 
//! state, fcStateConfig. Refer to the documentation of the message buffer
//! structures for further details.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param bufferId <br> [IN] The unique id of the message buffer which should
//!        be reconfigured. 
//! \param cfg [IN] <br> Message buffer configuration parameters.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information. 
//! \remarks Reconfiguring a consisting startup/sync transmit buffers with 
//!          message buffer type fcMsgBufNone deletes the buffer and it cannot
//!          be reconfigured again. 
EXPORTDLL fcError FCBASE_API fcbReconfigureMessageBufferSelfSynchronization(fcHandle hFlexCard, 
																			fcDword bufferId,
																			fcMsgBufCfg cfg);
  
//! This function reads a specific additional startup/sync message buffer 
//! configuration.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param bufferId [IN] <br> The identifier of the message buffer to be
//!        read.
//! \param cfg [OUT] <br> The configuration parameters of the specific message
//!        buffer.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
EXPORTDLL fcError FCBASE_API fcbGetCcMessageBufferSelfSynchronization(fcHandle hFlexCard, 
																	  fcDword bufferId, 
																	  fcMsgBufCfg* cfg);

//! This function resets the communication controller message buffers. The 
//! message buffers have no configuration (fcMsgBufNone). Since they also 
//! have no bufferId yet, the configuration cannot be read by 
//! fcbGetCcMessageBufferSelfSynchronization.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
EXPORTDLL fcError FCBASE_API fcbResetCcMessageBufferSelfSynchronization(fcHandle hFlexCard);

//! This function re-initializes the message buffer configuration of the self-startup
//! synchronization communication controller. After calling this function the
//! communication controller does not send old payload data. Re-initialization of message
//! buffers is only allowed if the communication controller is in configuration state.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
EXPORTDLL fcError FCBASE_API fcbReinitializeCcMessageBufferSelfSynchronization(fcHandle hFlexCard);

//! This function writes a self startup/sync data frame into a transmit buffer of
//! the FlexCard. The frame should normally be transmitted in the next
//! cycle. If the transmit acknowledgment is activated, an acknowledge packet is
//! generated as soon as the frame has been transmitted. This function should
//! only be called when the FlexCard is in normal active state or when all
//! message buffer configuration has been done.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param bufferId [IN] <br> The id of the message buffer used for the
//!        transmission.
//! \param payload [IN] The payload data to be transmitted.
//! \param payloadLength [IN] <br> The size of the payload data (number of
//!        2-byte words)
//! \remarks The payload data has to be organized as follows: if Data0 is the
//!          first byte to transmit and Data1 the second byte to transmit, then
//!          the high byte (Bit 8 - 15) of payload[0] contains Data1, the low
//!          byte (Bit0-7) of payload[0] contains Data0, etc.
//!          -----------------------------            
//!          |   Word 0  |   Word 1  | ...                
//!          -----------------------------                
//!          |Data1|Data0|Data3|Data2| ...                
//!          -----------------------------                
//!
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
EXPORTDLL fcError FCBASE_API fcbTransmitSelfSynchronization(fcHandle hFlexCard,
															fcDword bufferId,
															fcWord payload[],
															fcByte payloadLength);

//! This function writes a persistent user ID to the FlexCard.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param UserDefinedCardId [IN] <br> The ID that will be given to the FlexCard.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
EXPORTDLL fcError FCBASE_API fcbSetUserDefinedCardId(fcHandle hFlexCard,
													 fcDword UserDefinedCardId);

//! This function reads the persistent ID from the FlexCard.
//! \param hFlexCard [IN] <br> Handle to a FlexCard
//! \param pUserDefinedCardId [OUT] <br> The user defined FlexCard ID.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
EXPORTDLL fcError FCBASE_API fcbGetUserDefinedCardId(fcHandle hFlexCard,
													 fcDword* pUserDefinedCardId);

#endif /* #define _FCBASE_H_ */
