// Linux libraries
#include <unistd.h>
#include <string.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#include <fstream>
#include <sys/mman.h>

// Eberspaecher Electronics libraries
#include "event.h"
#include "Util.h"
#include <fcBase/fcBase.h>
#include <fcBase/fcBaseFlexRay.h>
#include <fcBase/fcBaseCAN.h>
#include <fcBase/fcBasePMC.h>
#include <fcBase/fcBaseLinux.h>

// Defines
#define TXFRAMECOUNT	2
#define MAXCCCOUNT		2

#define ACTIVATETXACKS	1	// 0: No TxFrames are generated - 1: Full TxFrame are generated
#define USETRIGGER		1	// 0: No Triggers are generated - 1: Start of cycle triggers are generated
#define USEEVENTSEM		1	// 0: Polling - 1: Use CycleStartEvent

// Transmit Frame
typedef struct transmitFrame
{
	fcDword FrameId;
	fcDword BufferIdx;
	fcWord Payload[fcPayloadMaximum];
	fcByte PayloadLength;
} transmitFrame;

// Prototyps
void printError(const char* szFunction, fcError e, int CcIndex = -1);
void printFlexCards(fcInfoHwSw* pInfo);
void printInfoHw(fcInfoHw* pInfoHw);
void printInfoSw(fcInfoSw* pInfoSw);
void printVersion(const fcVersionNumber& v);
void ProcessPackets(fcPacket* pPackets);
void CleanUp(void);

// Variables
unsigned int read_count = 0;
fcHandle hFlexCard = NULL;
fcFlexCardDeviceId gDevId = fcNoDevice;
transmitFrame tx_frame[TXFRAMECOUNT];

// fcDemoSigHandler()
static void fcDemoSigHandler(int signal)
{
	// flush any pending printf's
	fflush(stdout);
	// clean up the application
	CleanUp();
}

// main()
int main(int argc, char *argv[])
{
	unsigned int flexCardId = 0, ccFlexRay = 0;
	fcInfoHwSw* pInfo = NULL;
	fcError e = 0;
	char *pTemp = NULL, *UserId = NULL;
	EventSem eventCC1;

	// FlexCard requires ROOT user
	if (0 != strcmp((UserId = cuserid(NULL)), "root"))
	{
		printf("User %s must run as root\n", UserId);
		return -1;
	}

	if (mlockall(MCL_CURRENT | MCL_FUTURE))
	{
		printf("mlockall() failed\n");
		return -1;
	}

	// Initialize application signal handler
	signal(SIGINT, fcDemoSigHandler);

	// Check application parameters
	switch (argc)
	{
		case 3:
			break;
		case 4:
			read_count = strtoul(argv[3], &pTemp, 0);
			break;
		default:
			printf("\nUsage: ./fcDemoLinux [CHI-File1] [CHI-File2] [optional: Duration (Cycle)]\n\n");
			CleanUp();
	}

	std::string chiContent[MAXCCCOUNT];
	std::ifstream file1(argv[1]);
	std::ifstream file2(argv[2]);

	// Read the CHI-Files
	if (!file1.is_open())
	{
		printf("Could not open chi-file1: %s\n", argv[1]);
		CleanUp();
	}
	else
	{
		char ch;
		while (file1.get(ch))
			chiContent[fcCC1] += ch;
		file1.close();
	}

	if (!file2.is_open())
	{
		printf("Could not open chi-file2: %s\n", argv[2]);
		CleanUp();
	}
	else
	{
		char ch;
		while (file2.get(ch))
			chiContent[fcCC2] += ch;
		file2.close();
	}

	pInfo = NULL;
	e = fcbGetEnumFlexCardsV3(&pInfo, false); // get only available FlexCards
	if (0 != e)
	{
		printError("fcbGetEnumFlexCardsV3", e);
		CleanUp();
	}

	// save the first available FlexCard identifier, type and FlexRay counts
	flexCardId = pInfo->FlexCardId;
	gDevId = pInfo->InfoHardware.DeviceId;
	ccFlexRay = pInfo->InfoHardware.UseableCCs.FlexRay;

	// print out all installed FlexCards
	printf("\n----- Eberspaecher Electronics - fcDemoLinux -----\n\n");
	printFlexCards(pInfo);

	// free up allocated memory
	e = fcFreeMemory(fcMemoryTypeInfoHwSw, pInfo);
	if (0 != e)
	{
		printError("[fcFreeMemory]", e);
		CleanUp();
	}

	if (0 == flexCardId)
	{
		printf("No FlexCard available.\n");
		CleanUp();
	}

	if ((fcFlexCardPMC != gDevId && fcFlexCardPMCII != gDevId) || 2 > ccFlexRay)
	{
		printf("This demo application runs with FlexCard PMC (II) with 2 FlexRay-CCs only.\n");
		CleanUp();
	}

	// initialize Transmit-Frames
	memset(tx_frame, 0, sizeof(transmitFrame) * TXFRAMECOUNT);

	e = fcbOpen(&hFlexCard, flexCardId);
	if (0 != e)
	{
		printError("[fcbOpen]", e);
		CleanUp();
	}

	for (int CcIndex = fcCC1; CcIndex < MAXCCCOUNT; CcIndex++)
	{
		// Configure both CCs with chi strings
		e = fcbFRSetCcConfigurationChi(hFlexCard, (fcCC) CcIndex, chiContent[CcIndex].c_str());
		if (0 != e)
		{
			printError("[fcbFRSetCcConfigurationChi]", e, CcIndex);
			CleanUp();
		}

		// the first valid buffer is 1
		unsigned int bufIdx = 1;
		while (true)
		{
			fcMsgBufCfg cfg;
			memset(&cfg, 0, sizeof(fcMsgBufCfg));

			// try to get each buffer as long no error occurs
			if (0 == fcbFRGetMessageBuffer(hFlexCard, (fcCC) CcIndex, bufIdx, &cfg))
			{
				if (cfg.Type == fcMsgBufTx)
				{
					if (((fcCC1 == CcIndex) && (6 == cfg.Tx.FrameId)) ||
					    ((fcCC2 == CcIndex) && (8 == cfg.Tx.FrameId)))
					{
						// StatFrame 6 or 8 Channel A
						tx_frame[CcIndex].FrameId = cfg.Tx.FrameId;
						tx_frame[CcIndex].BufferIdx = bufIdx;
						tx_frame[CcIndex].PayloadLength = cfg.Tx.PayloadLength;
#if (1 == ACTIVATETXACKS)
						cfg.Tx.TxAcknowledgeEnable = 1;
						cfg.Tx.TxAcknowledgeShowNullFrames = 1;
						cfg.Tx.TxAcknowledgeShowPayload = 1;

						e = fcbFRReconfigureMessageBuffer(hFlexCard, (fcCC) CcIndex, bufIdx, cfg);
						if (0 != e)
						{
							printError("[fcbFRReconfigureMessageBuffer]", e, CcIndex);
							CleanUp();
						}
#endif
						break;
					}
				}
			}
			else
				break;

			bufIdx++;
		}
	}

	if ((6 != tx_frame[fcCC1].FrameId) || (8 != tx_frame[fcCC2].FrameId))
	{
		printf("Couldn't found transmit buffers!\n");
		CleanUp();
	}

	// set FlexRay bus termination
	fcbSetBusTermination(hFlexCard, fcBusChannel1, fcBusTypeFlexRay, true);
	fcbSetBusTermination(hFlexCard, fcBusChannel2, fcBusTypeFlexRay, false);
	fcbSetBusTermination(hFlexCard, fcBusChannel3, fcBusTypeFlexRay, true);
	fcbSetBusTermination(hFlexCard, fcBusChannel4, fcBusTypeFlexRay, false);

#if (1 == USETRIGGER)
	// Generate a pulse on trigger line 1, when the cycle start of cc1 occurs
	fcTriggerConfigurationEx triggerCfg;
	memset(&triggerCfg, 0, sizeof(fcTriggerConfigurationEx));
	triggerCfg.Condition = fcTriggerPMCOutOnCycleStart;
	triggerCfg.TriggerLineToConfigure = 1;
	triggerCfg.TriggerGeneratingCC = fcCC1;
	e = fcbSetTrigger(hFlexCard, triggerCfg);
	if (0 != e)
	{
		printError("[fcbSetTrigger]", e);
		CleanUp();
	}

	// Generate a trigger packet when a pulse on trigger line 2 is detected
	memset(&triggerCfg, 0, sizeof(fcTriggerConfigurationEx));
	triggerCfg.Condition = fcTriggerPMCIn;
	triggerCfg.onEdge = 0;	// 0: falling edge - 1: rising edge
	triggerCfg.TriggerLineToConfigure = 2;
	e = fcbSetTrigger(hFlexCard, triggerCfg);
	if (0 != e)
	{
		printError("[fcbSetTrigger]", e);
		CleanUp();
	}
#endif

#if (1 == USEEVENTSEM)
	// set event handle and start monitoring CC1
	e = fcbSetEventHandleSemaphore(hFlexCard, fcCC1, (void *) eventCC1, fcNotificationTypeFRCycleStarted);
	if (0 != e)
	{
		printError("[fcbSetEventHandleSemaphore]", e, fcCC1);
		CleanUp();
	}

	e = fcbFRMonitoringStart(hFlexCard, fcCC1, fcMonitoringNormal, true, true, true, false);
#else
	e = fcbFRMonitoringStart(hFlexCard, fcCC1, fcMonitoringNormal, true, false, true, false);
#endif
	if (0 != e)
	{
		printError("[fcbFRMonitoringStart]", e, fcCC1);
		CleanUp();
	}

	// Start monitoring CC2
	e = fcbFRMonitoringStart(hFlexCard, fcCC2, fcMonitoringNormal, false, false, true, false);
	if (0 != e)
	{
		printError("[fcbFRMonitoringStart]", e, fcCC2);
		CleanUp();
	}

	bool bStopMonitoring = false;
	unsigned int i = 0;
	do
	{
#if (1 == USEEVENTSEM)
		// Wait for NewCycle on CC1
		fcDword nTimeoutInMilliSeconds = 10000;
		bool bRet = eventCC1.WaitForEvent(nTimeoutInMilliSeconds);
		if (!bRet)
		{
			printError("[WaitForEvent]", 0);
			CleanUp();
		}
#endif
		
		// Receive all frames
		fcPacket* pPackets = NULL;
		e = fcbReceive(hFlexCard, &pPackets);
		if (0 != e)
		{
			printError("[fcbReceive]", e);
			CleanUp();
		}
		else
		{
			if (NULL != pPackets)
			{
				ProcessPackets(pPackets);
				e = fcFreeMemory(fcMemoryTypePacket, pPackets);
				if (0 != e)
				{
					printError("[fcFreeMemory]", e);
					CleanUp();
				}
			}
#if (0 == USEEVENTSEM)
			else
			{
				// Polling mode, wait 1 ms
				ms_sleep(1);
			}
#endif
		}

		// Transmit frames
		for (int CcIndex = fcCC1; CcIndex < MAXCCCOUNT; CcIndex++)
		{
			e = fcbFRTransmit(hFlexCard, (fcCC) CcIndex, tx_frame[CcIndex].BufferIdx, tx_frame[CcIndex].Payload, tx_frame[CcIndex].PayloadLength);
			if (0 != e)
			{
				printError("[fcbFRTransmit]", e, CcIndex);
				printf("Error: Frame %d\n", tx_frame[CcIndex].FrameId);
			}
		}

		// Update transmit frames
		for (int CcIndex = fcCC1; CcIndex < MAXCCCOUNT; CcIndex++)
		{
			//////////////////////////////////////////////////
			// Ramp signal
			if (tx_frame[CcIndex].Payload[0] < 50000)
				tx_frame[CcIndex].Payload[0] += 100;
			else
				tx_frame[CcIndex].Payload[0] = 0;
		}

		if (0 != read_count)
		{
			i++;
			if (i >= read_count)
				bStopMonitoring = true;
		}
	} while (! bStopMonitoring);

	CleanUp();

	return 0;
}

// CleanUp()
void CleanUp()
{
	if (NULL != hFlexCard)
	{
		fcError e = 0;

		// Stop monitoring CC1 and CC2
		for (int CcIndex = fcCC1; CcIndex < MAXCCCOUNT; CcIndex++)
		{
			e = fcbFRMonitoringStop(hFlexCard, (fcCC) CcIndex);
			if (0 != e)
				printError("[fcbFRMonitoringStop]", e, CcIndex);
		}

		// reset FlexRay bus termination
		fcbSetBusTermination(hFlexCard, fcBusChannel1, fcBusTypeFlexRay, false);
		fcbSetBusTermination(hFlexCard, fcBusChannel2, fcBusTypeFlexRay, false);
		fcbSetBusTermination(hFlexCard, fcBusChannel3, fcBusTypeFlexRay, false);
		fcbSetBusTermination(hFlexCard, fcBusChannel4, fcBusTypeFlexRay, false);

		// Close FlexCard
		e = fcbClose(hFlexCard);
		if (0 != e)
			printError("[fcbClose]", e);

		hFlexCard = NULL;
	}

	// flush any pending printf's
	fflush(stdout);

	munlockall();

	exit(0);
}

void printError(const char* szFunction, fcError e, int CcIndex)
{
	char* szError = NULL;
	fcError error = fcGetErrorText(e, &szError);
	if (0 != error)
	{
		printf("printError: Function fcGetErrorText returned with error 0x%X\n", e);
	}
	else
	{
		if (0 >= CcIndex)
			printf("Error: Function %s (CC%01d) returned with error 0x%X (%s)\n", szFunction, CcIndex+1, e, szError);
		else
			printf("Error: Function %s returned with error 0x%X (%s)\n", szFunction, e, szError);
	}

	if (NULL != szError)
	{
        error = fcFreeMemory(fcMemoryTypeString, szError);
		if (0 != error)
			printf("printError: Function fcFreeMemory returned with error 0x%X\n", e);
	}
}

void printVersion(const fcVersionNumber& v)
{
	printf("%d.%d.%d.%d\n", v.Major, v.Minor, v.Update, v.Release);
}

void printFlexCards(fcInfoHwSw* pInfo)
{
	while (pInfo != NULL)
	{
		// if FlexCard Id is equal to zero we got NO FlexCard in the system
		bool bFlexCardAvailable = (0 != pInfo->FlexCardId);
		
		printf("FlexCard Id:  ");
		if (bFlexCardAvailable)
			printf("%u\n", pInfo->FlexCardId);
		else
			printf("Not available\n");

		printf("UserCard Id:  ");
		if (bFlexCardAvailable)
			printf("%u\n", pInfo->UserDefinedCardId);
		else
			printf("Not available\n");

		printf("Dev Busy:     ");
		if (bFlexCardAvailable)
			printf("%d\n", pInfo->Busy);
		else
			printf("Not available\n");

		if (bFlexCardAvailable)
		{
			printInfoHw(&pInfo->InfoHardware);
			printInfoSw(&pInfo->InfoSoftware);
		}
		else
		{
			printf("Software Information:\n");

			printf("\tDLL:      ");
			printVersion(pInfo->InfoSoftware.VersionBaseDll);

			printf("\tDriver:   ");
			printVersion(pInfo->InfoSoftware.VersionDeviceDriver);
		}

		printf("\n");

		pInfo = pInfo->pNext; // get the next FlexCard
	}
}

void printInfoHw(fcInfoHw* pInfoHw)
{
	if (NULL == pInfoHw) return;
	printf("Hardware Information:\n");

	printf("\tSerial:   %llu\n", pInfoHw->Serial);

	printf("\tDevId:    ");
	switch (pInfoHw->DeviceId)
		{
			case fcFlexCardCycloneII:
				printf("Cyclone II\n");
				break;
			case fcFlexCardCycloneIISE:
				printf("Cyclone II SE\n");
				break;
			case fcFlexCardPMC:
				printf("PMC\n");
				break;
		case fcFlexCardPMCII:
			printf("PMCII\n");
			break;
			default:
				printf("not available\n");
				break;
		};

	printf("\tFirmware: ");
	printVersion(pInfoHw->VersionFirmware);

	printf("\tHardware: ");
	printVersion(pInfoHw->VersionHardware);
		
	printf("\t-BusType- Sup Lic Use\n");
	printf("\t FlexRay   %1d   %1d   %1d\n",
		pInfoHw->SupportedCCs.FlexRay, pInfoHw->LicensedCCs.FlexRay, pInfoHw->UseableCCs.FlexRay);
	printf("\t CAN       %1d   %1d   %1d\n",
		pInfoHw->SupportedCCs.CAN, pInfoHw->LicensedCCs.CAN, pInfoHw->UseableCCs.CAN);

	// TODO: Output for 'pInfoHw->pVersionCC'
}

void printInfoSw(fcInfoSw* pInfoSw)
{
	if (NULL == pInfoSw) return;
	printf("Software Information:\n");

	printf("\tDLL:      ");
	printVersion(pInfoSw->VersionBaseDll);

	printf("\tDriver:   ");
	printVersion(pInfoSw->VersionDeviceDriver);

	printf("\tLicense:  Windows %d Linux %d Xenomai %d\n",
		pInfoSw->LicensedForWindowsDriver, pInfoSw->LicensedForLinuxDriver, pInfoSw->LicensedForXenomaiDriver);
}

void ProcessPackets(fcPacket* pPackets)
{
	fcPacket* p = pPackets;
	while (NULL != p)
	{
		switch (p->Type)
		{
			case fcPacketTypeInfo:
			case fcPacketTypeFlexRayFrame:
			case fcPacketTypeError:
			case fcPacketTypeStatus:
			case fcPacketTypeTrigger:
			case fcPacketTypeTxAcknowledge:
			case fcPacketTypeNMVector:
			case fcPacketTypeNotification:
			case fcPacketTypeTriggerEx:
			case fcPacketTypeCAN:
			case fcPacketTypeCANError:
				Util::print(p, gDevId);
				break;
			default:
				break;
		};

		// get the next packet
		p = p->pNextPacket;
	}
}
