/*/////////////////////////////////////////////////////////////////////////////
//! \file fcBaseFlexRay.h:
//! \brief Header File for the fcBase FlexRay API.
//!
//! Use this header file for developing user defined FlexCard applications.
//! See also printed documentation for more information. The common data types
//! and enumerations can be found in the header file "fcBaseTypes.h".
//!
//! Generally:
//!
//! - Each function of this library (except some error handling functions)
//!   returns an error code. If the return value is equal to 0, no error
//!   occured. A number greater than 0 indicates an error. To get more
//!   information about it, use the error handling functions.
//!
//! \par Overview:
//! Author:         EBEL/WO/SH/Ba/TB/Tr/LJ
//! $Date: 2009-01-29 10:04:25 +0100 (Do, 29 Jan 2009) $ <br>
//! \par Copyright:
//! Copyright 2008-2009 Eberspaecher Electronics GmbH & Co. KG
//! \version $Revision: 14916 $
/////////////////////////////////////////////////////////////////////////////*/

#ifndef _FCBASE_FLEXRAY_H_
#define _FCBASE_FLEXRAY_H_

#ifndef _FCBASE_TYPES_H_
	#error Please include first the header files "fcBaseTypes.h"
#endif

#ifndef _FCBASE_TYPES_FLEXRAY_H_
	#error Please include first the header files "fcBaseTypesFlexRay.h"
#endif

// Linux
#ifdef linux

	#ifndef FCBASE_API
		#define FCBASE_API
	#endif
	//! Please do not use the FCBASE_EXPORTS define in your application, unless
	//! you have the source code ;-)
	#ifdef FCBASE_EXPORTS
		#define EXPORTDLL extern "C" __attribute__ ((visibility ("default")))
	#else
		#if defined(__cplusplus)
			#define EXPORTDLL extern "C"
		#else
			#define EXPORTDLL
		#endif
	#endif

// Windows
#else

	#ifndef FCBASE_API
		#define FCBASE_API __stdcall
	#endif
	//! Please do not use the FCBASE_EXPORTS define in your application, unless
	//! you have the source code ;-)
	#ifdef FCBASE_EXPORTS
		#define EXPORTDLL extern "C" __declspec(dllexport)
	#else
		#if defined(__cplusplus)
			#define EXPORTDLL extern "C" __declspec(dllimport)
		#else
			#define EXPORTDLL __declspec(dllimport)
		#endif
	#endif

#endif


//! This function configures the FlexRay communication controller of the FlexCard
//! with a FlexConfig compatible configuration string (CHI File). The payload data
//! for transmit message buffers are not set by this function. Before the
//! configuration of the FlexRay CC starts, all message buffers are reset to their
//! default settings (FIFO buffer).
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param szChi [IN] <br> Pointer to null-terminated CHI string (refer to the
//! 	   CHI string example section).
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \note Internally, the function uses the #fcbFRSetCcRegister function,
//! 	  therefore the same restrictions as for writing register exist.
//! \see #fcCC
EXPORTDLL fcError FCBASE_API fcbFRSetCcConfigurationChi(fcHandle hFlexCard,
														fcCC CC,
														const char* szChi);

//! This function configures the FlexRay communication controller of the FlexCard
//! with a CANdb compatible string. Before the configuration of the FlexRay CC
//! starts, all message buffers are reset to their default settings (FIFO buffer).
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param szCANdb [IN] <br> Pointer to null-terminated CANdb string.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \note Internally, the function uses the #fcbFRSetCcRegister function; therefore
//! 	  the same restrictions as for writing register exist.
//! \see #fcCC
EXPORTDLL fcError FCBASE_API fcbFRSetCcConfigurationCANdb(fcHandle hFlexCard,
														  fcCC CC,
														  const char* szCANdb);

//! This function configures the FlexRay communication controller of the FlexCard
//! with a #fcFRCcConfig structure. The payload data for transmit message buffers
//! are not set by this function. Before the configuration of the FlexRay CC starts,
//! all message buffers are reset to their default settings (FIFO buffer).
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param cfg [IN] <br> FlexRay communication controller configuration.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \note Internally, the function uses the #fcbFRSetCcRegister function,
//! 	  therefore the same restrictions as for writing register exist.
//! \see #fcCC, #fcFRCcConfig, #fcbFRGetCcConfiguration
EXPORTDLL fcError FCBASE_API fcbFRSetCcConfiguration(fcHandle hFlexCard,
													 fcCC CC,
													 fcFRCcConfig cfg);

//! This function reads the FlexRay communication controller configuration of
//! the FlexCard.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param pCfg [OUT] <br> The configuration parameters of the FlexRay
//!        communication controller.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \note Internally, the function uses the #fcbFRGetCcRegister function,
//! 	  therefore the same restrictions as for writing register exist.
//! \see #fcCC, #fcFRCcConfig, #fcbFRSetCcConfiguration
EXPORTDLL fcError FCBASE_API fcbFRGetCcConfiguration(fcHandle hFlexCard,
													 fcCC CC,
													 fcFRCcConfig* pCfg);

//! This function configures transmit, receive and FIFO message buffers of the
//! FlexRay communication controller. Configuring message buffers is only allowed
//! if the FlexRay CC is in its configuration state, fcStateConfig.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param pBufferId <br> [OUT] Message buffer identifier. If the configuration
//! 	   was successful the message buffer identifier is greater than 0. This
//! 	   identifier will be required to transmit the content of the buffer (in
//! 	   the case of a transmit buffer).
//! \param cfg [IN] <br> Message buffer configuration parameters.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \remarks Before configuring the message buffers, it is necessary to set up
//! 		 the global FlexRay communication parameters (cluster parameters).
//! 		 Internally the FlexCard used the FIFO buffers as receive buffers,
//! 		 therefore we recommend using FIFO message buffers as much as
//! 		 possible.
//! \see #fcCC, #fcMsgBufCfg
EXPORTDLL fcError FCBASE_API fcbFRConfigureMessageBuffer(fcHandle hFlexCard,
														 fcCC CC,
														 fcDword* pBufferId,
														 fcMsgBufCfg cfg);

//! This function reads a FlexRay specific message buffer configuration.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param bufferId [IN] <br> The identifier of the message buffer to be read.
//! \param pCfg [OUT] <br> The configuration parameters of the specific message
//!        buffer.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \see #fcCC, #fcMsgBufCfg, #fcbFRReconfigureMessageBuffer
EXPORTDLL fcError FCBASE_API fcbFRGetMessageBuffer(fcHandle hFlexCard,
												   fcCC CC,
												   fcDword bufferId,
												   fcMsgBufCfg* pCfg);

//! This function reconfigures transmit, receive and FIFO message buffers of the
//! FlexRay communication controller. A reconfiguration is only allowed for message
//! buffers which are already configured. This function is available in all
//! states of the FlexRay CC. Not all configuration settings could be modified in
//! monitoring state. Refer to the documentation of the message buffer structures
//! for further details.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param bufferId <br> [IN] The unique id of the message buffer which should
//!        be reconfigured.
//! \param cfg [IN] <br> Message buffer configuration parameters.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \see #fcCC, #fcMsgBufCfg, #fcbFRGetMessageBuffer
EXPORTDLL fcError FCBASE_API fcbFRReconfigureMessageBuffer(fcHandle hFlexCard,
														   fcCC CC,
														   fcDword bufferId,
														   fcMsgBufCfg cfg);

//! This function resets the FlexRay communication controller message buffers.
//! After calling this function, all message buffers are configured as receive
//! FIFO - with maximal payload (depends on the FlexRay communication controller).
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \see #fcCC
EXPORTDLL fcError FCBASE_API fcbFRResetMessageBuffers(fcHandle hFlexCard,
													  fcCC CC);

//! This function configures the FlexRay frame ids accepted by the device driver.
//! Only the FlexRay ids which are in the filter list are forwarded to the user
//! application, all other FlexRay frames are rejected. To accept all frames set the
//! parameters pData to NULL and size to zero or configure a single frame id of zero.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param channel [IN] <br> FlexCard channel(s) concerned by the filter.
//! \param pData [IN] <br> Pointer to an fcDword array containing the ids which
//!        are accepted. Each element (fcDword) contains one frame identifier.
//!        --------------------------------
//!        |  fcDword  |  fcDword  | ...
//!        --------------------------------
//!        |   ID x    |   ID y    | ...
//!        --------------------------------
//! \param size [IN] <br> Number of ids in the array.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
//! \see #fcCC, #fcChannel, #fcbReceive
EXPORTDLL fcError FCBASE_API fcbFRSetSoftwareAcceptanceFilter(fcHandle hFlexCard,
															  fcCC CC,
															  fcChannel channel,
															  fcDword* pData,
															  fcDword size);

//! Sets the transceiver mode individually for a FlexRay channel.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param stateChA [IN] <br> The new transceiver state for channel A.
//! \param stateChB [IN] <br> The new transceiver state for channel B.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         please use the functions described in the section Error Handling to
//!         get extended error information.
//! \remarks If one of the transceiver is in the sleep mode and the transceiver
//! 		 detects a wakeup event, the notification event
//! 		 fcNotificationTypeFRWakeup is fired once only.
//! \see #fcCC, #fcTransceiverState, #fcbFRGetTransceiverState
EXPORTDLL fcError FCBASE_API fcbFRSetTransceiverState(fcHandle hFlexCard,
													  fcCC CC,
													  fcTransceiverState stateChA,
													  fcTransceiverState stateChB);

//! Gets the current transceiver state for a FlexRay channel.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param pStateChA [OUT] <br> The current transceiver state for channel A.
//! \param pStateChB [OUT] <br> The current transceiver state for channel B.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         please use the functions described in the section Error Handling to
//!         get extended error information.
//! \remarks If one of the transceiver is in the sleep mode and the transceiver
//! 		 detects a wakeup event, the notification event
//! 		 fcNotificationTypeFRWakeup is fired once only.
//! \see #fcCC, #fcTransceiverState, #fcbFRSetTransceiverState
EXPORTDLL fcError FCBASE_API fcbFRGetTransceiverState(fcHandle hFlexCard,
													  fcCC CC,
													  fcTransceiverState* pStateChA,
													  fcTransceiverState* pStateChB);

//! This function writes a value in a given register of the FlexRay communication
//! controller. Not every register can be written (e.g. the registers belonging
//! to the message buffer configuration or some interrupt settings).
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param address [IN] <br> Address of the CC register to be written.
//! \param value [IN] <br> The value to be written.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         please use the functions described in the section Error Handling to
//!         get extended error information. If the register can not be written
//!         the error code REGISTER_NOT_WRITEABLE is returned
//! \remarks For a register description refer to the specification of the
//!          corresponding FlexRay communication controller. Modifying one of the
//!          following registers will reset message buffers with their default
//!          settings (FIFO receive buffers). The users message buffers.
//!          configuration will not be valid anymore:
//!          - Bosch E-Ray: MHDC (0x0098) and GTUC7 (0x00B8).
//! \note: Not all register of a communication controller can be
//!        set. The base API will modify some parameters so that the operating
//!        of the FlexCard is guaranteed (e.g. interrupt settings). Access is
//!        denied to all register which are used for FlexRay message buffer
//!        configuration.
//! \see #fcCC, #fcbFRGetCcRegister
EXPORTDLL fcError FCBASE_API fcbFRSetCcRegister(fcHandle hFlexCard,
												fcCC CC,
												fcDword address,
												fcDword value);

//! This function reads and returns the content of a given register of the
//! FlexRay communication controller.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param address [IN] <br> Address of the CC register to be read.
//! \param pValue [OUT] <br> The content of the desired FlexRay CC register.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information. If the register can not be read the
//!         error code REGISTER_NOT_READABLE is returned.
//! \remarks Not every register could be read. For a register description refer
//!          to the specification of the corresponding FlexRay CC.
//! \see #fcCC, #fcbFRSetCcRegister
EXPORTDLL fcError FCBASE_API fcbFRGetCcRegister(fcHandle hFlexCard,
												fcCC CC,
												fcDword address,
												fcDword* pValue);

//! This function returns the current FlexRay communication controller POC state.
//! For a description of possible state, refer to the enumeration #fcState. This
//! function should be used to check if the integration into a FlexRay cluster
//! has succeeded.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param pState [OUT] <br> Current FlexRay communication controller state.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         please use the functions described in the section Error Handling to
//!         get extended error information.
//! \see #fcCC, #fcbFRMonitoringStart, #fcbFRMonitoringStop
EXPORTDLL fcError FCBASE_API fcbFRGetCcState(fcHandle hFlexCard,
											 fcCC CC,
											 fcState* pState);

//! This function is used to start the monitoring of a FlexRay bus. Once called,
//! the function changes the FlexRay CC state from configuration state to normal
//! active state (if the cluster integration succeeds). The current FlexRay
//! communication controller state can be read using the function
//! #fcbFRGetCcState. If the FlexCard is synchronized with the cluster the
//! function #fcbFRGetCcState will return the value #fcStateNormalActive.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param mode [IN] <br> The monitoring mode. Not every monitoring mode is
//!        supported by all communication controllers. See #fcMonitoringModes
//! 	   for details
//! \param restartTimestamps [IN] <br> Set this parameter to true to restart the
//! 	   measurement without resetting the FlexCard timestamp. Set it to false
//! 	   to start the measurement from the beginning. The timestamps have
//! 	   micro second resolution.
//! \param enableCycleStartEvents [IN] <br> Set this parameter to true to enable
//! 	   the cycle start events in order that at the beginning of every cycle
//! 	   the event fcNotificationTypeFRCycleStarted is signalled.
//! \param enableColdstart [IN] <br> Set this parameter to true to allow the
//! 	   FlexCard to initialize the cluster communication, otherwise the
//! 	   coldstart inhibit mode is active. This feature can not be used in the
//! 	   monitoring mode fcMonitoringDebug and fcMonitoringDebugAsynchron.
//! \param enableWakeup [IN] <br> Set this parameter to true to transmit a
//! 	   wakeup pattern on the configured wakeup channel (FlexRay Protocol
//! 	   Specification: pWakeupChannel). A cluster wakeup must precede the
//! 	   communication start up to ensure that all nodes in a cluster are
//! 	   awake. The minimum requirement for a cluster wakeup is that all bus
//! 	   drivers are supplied with power. This feature can not be used in the
//! 	   monitoring mode fcMonitoringDebug and fcMonitoringDebugAsynchron.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         please use the functions described in the section Error Handling to
//!         get extended error information.
//! \remarks After the monitoring has started, the user should check if the
//!          integration in the cluster was successful, #fcbFRGetCcState should
//!          return the state #fcStateNormalActive.
//! \note After the monitoring has successfully started, the receive process has
//! 	  to be started as soon as possible to avoid an overflow (error packet
//! 	  fcErrFlexcardOverflow is received). Once an overflow occurred, no more
//! 	  packets can be received. The monitoring has to be stopped and started
//! 	  again.
//! \see #fcCC, #fcMonitoringModes, #fcbFRMonitoringStop, #fcbFRGetCcState,
//!      #fcbSetEventHandleV2, #fcbSetEventHandleSemaphore
EXPORTDLL fcError FCBASE_API fcbFRMonitoringStart(fcHandle hFlexCard,
												  fcCC CC,
												  fcMonitoringModes mode,
												  fcBool restartTimestamps,
												  fcBool enableCycleStartEvents,
												  fcBool enableColdstart,
												  fcBool enableWakeup);

//! This function stops the FlexRay bus measurement. The FlexRay communication
//! controller is set back in its configuration state.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
//! \see #fcCC, #fcbMonitoringStart
EXPORTDLL fcError FCBASE_API fcbFRMonitoringStop(fcHandle hFlexCard,
												 fcCC CC);

//! This function writes a data frame into a FlexRay communication controller
//! transmit buffer of the FlexCard. The frame should normally be transmitted
//! in the next cycle. If the transmit acknowledgment is activated, an acknowledge
//! packet is generated as soon as the frame has been transmitted. This function
//! should only be called when the FlexCard is in normal active state or when all
//! message buffer configuration has been done.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param bufferId [IN] <br> The id of the message buffer used for the
//!        transmission.
//! \param pPayload [IN] The payload data to be transmitted.
//! \param payloadLength [IN] <br> The size of the payload data (number of
//!        2-byte words)
//! \remarks The payload data has to be organized as follows: if Data0 is the
//!          first byte to transmit and Data1 the second byte to transmit, then
//!          the high byte (Bit 8 - 15) of payload[0] contains Data1, the low
//!          byte (Bit0-7) of payload[0] contains Data0, etc.
//!          -----------------------------
//!          |   Word 0  |   Word 1  | ...
//!          -----------------------------
//!          |Data1|Data0|Data3|Data2| ...
//!          -----------------------------
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
//! \see #fcCC
EXPORTDLL fcError FCBASE_API fcbFRTransmit(fcHandle hFlexCard,
										   fcCC CC,
										   fcDword bufferId,
										   fcWord* pPayload,
										   fcByte payloadLength);

//! This function transmits a symbol in the FlexRay symbol window segment. It
//! can only be called if the FlexRay communication controller is in the POC
//! state NORMAL_ACTIVE. For a list of available symbols to be transmitted,
//! see the enumeration #fcSymbolType.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param type [IN] <br> Type of symbol to transmit.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
//! \see #fcCC, #fcSymbolType
EXPORTDLL fcError FCBASE_API fcbFRTransmitSymbol(fcHandle hFlexCard,
												 fcCC CC,
												 fcSymbolType type);

//! This function calculates the macrotick offset for a specific cycle
//! position in a FlexRay cycle.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param CyclePosition [IN] <br> The FlexRay cycle position.
//! \param slotId [IN] <br> This parameter is used for a FlexRay cycle
//!	   position of fcCyclePosStaticSlot and fcCyclePosDynamicMiniSlot
//!	   to calculate the macrotick offset for a static slot or a dynamic
//!	   mini slot id.
//! \param pValue [OUT] <br> The macrotick offset value.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \see #fcCC, #fcCyclePos, #fcCcTimerCfg, #fcbFRSetCcTimerConfig
EXPORTDLL fcError FCBASE_API fcbFRCalculateMacrotickOffset(fcHandle hFlexCard,
														   fcCC CC,
														   fcCyclePos CyclePosition,
														   fcDword slotId,
														   fcDword* pValue);

//! This function configures the FlexRay communication controller Timer0 interrupt.
//! To get a notification when the FlexRay CC Timer0 interval has occurred, an
//! event of type #fcNotificationTypeFRCcTimer has to be registered by the function
//! #fcbSetEventHandleV2 or #fcbSetEventHandleSemaphore. The FlexRay CC Timer0 can
//! be enabled/disabled by this function.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param cfg [IN] <br> The FlexRay CC Timer0 configuration.
//! \param bEnable [IN] <br> Set to true to enable the FlexRay CC Timer0, and
//! 	   to false to disable it.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
//! \see #fcCC, #fcCcTimerCfg, #fcbFRGetCcTimerConfig, #fcbSetEventHandleV2,
//!      #fcbSetEventHandleSemaphore
EXPORTDLL fcError FCBASE_API fcbFRSetCcTimerConfig(fcHandle hFlexCard,
												   fcCC CC,
												   fcCcTimerCfg cfg,
												   fcBool bEnable);

//! This function reads the FlexRay communication controller Timer0 configuration.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param pCfg [OUT] <br> The configuration parameters of the FlexRay CC Timer0.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \see #fcCC, #fcCcTimerCfg, #fcbFRSetCcTimerConfig
EXPORTDLL fcError FCBASE_API fcbFRGetCcTimerConfig(fcHandle hFlexCard,
												   fcCC CC,
												   fcCcTimerCfg* pCfg);

//! This function configures the FlexRay frame ids accepted by the FlexCard firmware.
//! Only the FlexRay ids which are in the filter list are forwarded to the device
//! driver, all other FlexRay frames are rejected. To accept all frames set the
//! parameters pData to NULL and size to zero or configure a single frame id of zero.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the FlexRay communication controller.
//! \param channel [IN] <br> FlexCard channel(s) concerned by the filter.
//! \param pData [IN] <br> Pointer to an fcDword array containing the ids which
//!        are accepted. Each element (fcDword) contains one frame identifier.
//!        --------------------------------
//!        |  fcDword  |  fcDword  | ...
//!        --------------------------------
//!        |   ID x    |   ID y    | ...
//!        --------------------------------
//! \param size [IN] <br> Number of ids in the array.
//! \param reset [IN] <br> Set to true to reset the filter, before configure a new filter.
//!        Set to false to add the frame identifier to the existing filter.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
//! \see #fcCC, #fcChannel, #fcbReceive
EXPORTDLL fcError FCBASE_API fcbFRSetHardwareAcceptanceFilter(fcHandle hFlexCard,
															  fcCC CC,
															  fcChannel channel,
															  fcDword* pData,
															  fcDword size,
															  fcBool reset);

#endif /* #define _FCBASE_FLEXRAY_H_ */
