/*/////////////////////////////////////////////////////////////////////////////
//! \file fcBaseTypes.h:
//! \brief Header file for the data types and enumerations of the fcBase API.
//!
//! Use this header file for developing user defined FlexCard applications. This
//! header file contains common structures, data types and enumerations which are
//! used in the fcBase API. For a description of the common API function, please
//! refer to the header file "fcBase.h".
//!
//! See also printed documentation for more information.
//!
//! \par Overview:
//! Author:         EBEL/WO/SH/Ba/TB/Tr/LJ
//! $Date: 2009-02-16 13:29:35 +0100 (Mo, 16 Feb 2009) $ <br>
//! \par Copyright:
//! Copyright 2008-2009 Eberspaecher Electronics GmbH & Co. KG
//! \version $Revision: 15009 $
/////////////////////////////////////////////////////////////////////////////*/

#ifndef _FCBASE_TYPES_H_
#define _FCBASE_TYPES_H_


//-----------------------------------------------------------------------------
//          Error handling
//-----------------------------------------------------------------------------


//! This type provides information about an error. A zero value means no error
//! occurred. To extract the detailed information about an error, use the
//! functions #fcGetErrorType, #fcGetErrorText and #fcGetErrorCode.
//! \note fcError is not equivalent to fcErrorCode. To get the fcErrorCode from
//!       a fcError you have to use the function fcGetErrorCode.
typedef unsigned int fcError;

//! This enumeration contains all error codes which are reported by the fcBase API functions. To
//! extract the error code from a #fcError use the #fcGetErrorCode function.
//! \see #fcGetErrorText
typedef enum fcErrorCode
{
    //! No error occurred
    NONE                = 0,
    INTERNAL_ERROR      = 99,
    //! Range 1 to 100 is reserved
    RESERVED            = 100,

    //
    // Global
    //

    //! One of the passed parameter is null
    NULL_PARAMETER,
    //! The passed parameter is not valid.
    INVALID_PARAMETER,
    //! The memory allocation failed.
    MEMORY_ALLOCATION_FAILED,
    //! The fcHandle isn't valid.
    INVALID_OBJECT_HANDLE,
    //! Couldn't perform action because this type of action isn't supported by
    //! the communication controller
    ACTION_NOT_SUPPORTED,
    //! Couldn't set this parameter because this parameter isn't supported by
    //! the communication controller
    PARAMETER_NOT_SUPPORTED,
    //! An action failed -> for details see error text
    ACTION_FAILED,
    //! ...
    ACTION_NOT_ALLOWED_DURING_MONITORING,

    //
    // Error handling
    //
    TEXT_NOT_DEFINIED,

    //
    // Initialization
    //

    //! A connection to the FlexCard is already open.
    CONNECTION_ALREADY_OPEN,
    //! The FlexCard belonging to this id doesn't exist.
    UNKOWN_FLEXCARD_ID,
    //! The monitoring is already active.
    MONITORING_IS_ALREADY_ACTIVE,


    //
    // Configuration
    //
    REGISTER_NOT_READABLE,
    REGISTER_NOT_WRITEABLE,
    //! Error while parsing string
    PARSING_ERROR,
    //! Incompatible configuration version
    INCOMPATIBLE_VERSION,
    //! Configuration failed
    CONFIGURATION_FAILED,

    RX_BUFFER_CANNOT_USED,
    INVALID_MSGBUF_SETTINGS,
    MINIMUM_FIFO_LIMIT,
    NOT_ENOUGH_MESSAGE_RAM,

    //! The given payload exceeds the maximum communication controller message
    //! buffer payload.
    PAYLOAD_EXCEEDS_MAXIMUM,

    //
    // Receive
    //

    //
    // Transmit
    //
    MSG_BUF_BUSY,

    // The message buffer is ready for transmission respectively transmission of
    // this message buffer is in progress. Update of this message buffer is
    // possible when the transmission is completed.
    MSG_BUF_LOCKED_FOR_TRANSMISSION,

    //! Incorrect POC state. The given operation expects another POC state.
    INCORRECT_POC_STATE,

	//! A firmware update is required
	FIRMWARE_UPDATE_REQUIRED,

	//! This function is not implemented yet
    FUNCTION_NOT_IMPLEMENTED,

	//! The connection was released when the FlexCard was removed (surprise removal).
	CONNECTION_RELEASED_AFTER_SURPRISE_REMOVAL,

	//! The CC index is not valid
	CC_INDEX_NOT_VALID,

	//! This function is for PMC cards only
	PMC_CARD_FUNCTION,

	//! This function can not be called cause another version of a trigger function was called before. You need can either use the previous function again or need to reopen the connection.
	TRIGGER_FUNCTION_VERSION_CONFLICT,

	//! The real-time task has been unblocked by a signal or explicitly via a task_unblock call.
	XENOMAI_TASK_UNBLOCKED,

	//! The event has been destroyed.
	XENOMAI_EVENT_DESTROYED,

	//! The event has not been satisfied within the specified amount of time.
	XENOMAI_TIMEDOUT,

	//! An illegal invocation environment was detected during the events wait phase.
	XENOMAI_ILLEGAL_INVOCATION,

	//! The function fcbWaitForEvent was called when not in real time context.
	XENOMAI_NO_REALTIME_CONTEXT,

	//! This function is for CARDBUS cards only.
	CARDBUS_CARD_FUNCTION,

	//! A driver update is required.
	DRIVER_UPDATE_REQUIRED,

	//! Cannot access any CAN CC on this FlexCard.
	CC_CAN_NOT_AVAILABLE,

	//! The versions of DLL, SYS and FW do not match.
	VERSION_NOT_MATCH,

	//! A parameter for the cc isn't valid.
	INVALID_CCCONFIG_SETTINGS,

	//! The bus type is not valid.
	BUS_TYPE_NOT_VALID,

	//! This firmware version doesn't support the self-synchronization feature.
	SELFSYNC_NOT_AVAILABLE,

	//! Cannot read CAN message buffer, because it has not been set yet.
	CAN_BUFFER_NOT_SET_YET,

	//! Invalid hardware license found, feature not available.
	INVALID_HARDWARE_LICENSE,

} fcErrorCode;


//! This enum defines the type of error information. To get the fcErrorType
//! from a #fcError, please use the #fcGetErrorType function.
//! \see #fcGetErrorText
typedef enum fcErrorType
{
    //! No error
    fcErrorTypeSuccess      = 0,

    //! The error should be treated as an information message that means the
    //! function has succeeded but wants to give additional information.
    fcErrorTypeInformation  = 1,

    //! The error should be treated as a warning message that means the function
    //! has succeeded but the input parameters are modified or not complete
    //! accepted.
    fcErrorTypeWarning      = 2,

    //! The error should be treated as an error message that means the function
    //! has failed.
    fcErrorTypeError        = 3,

} fcErrorType;


//-----------------------------------------------------------------------------
//          Memory handling
//-----------------------------------------------------------------------------

//! This enumeration defines the memory types needed to release the memory
//! allocated by the functions #fcGetErrorText, #fcbGetEnumFlexCards,
//! #fcbGetEnumFlexCardsV2, #fcbGetEnumFlexCardsV3, #fcbGetInfoFlexCard
//! and #fcbReceive.
//! \see #fcFreeMemory, #fcGetErrorText, #fcbGetEnumFlexCards, #fcbReceive,
//!		 #fcbGetEnumFlexCardsV2, #fcbGetEnumFlexCardsV3, #fcbGetInfoFlexCard
typedef enum fcMemoryType
{
	fcMemoryTypeString,		//!< Memory is of the type "char[]"
	fcMemoryTypeInfo,		//!< Memory is of the type "fcInfo"
	fcMemoryTypePacket,		//!< Memory is of the type "fcPacket"
	fcMemoryTypeInfoV2,		//!< Memory is of the type "fcInfoV2"
	fcMemoryTypeInfoHwSw,	//!< Memory is of the type "fcInfoHwSw"
} fcMemoryType;


//-----------------------------------------------------------------------------
//           Initialization
//-----------------------------------------------------------------------------


//! It defines a handle to a FlexCard object. A handle is returned by the
//! function #fcbOpen (assuming that a valid FlexCard identifier has been used).
typedef void* fcHandle;

//! Boolean value.
typedef unsigned char fcBool;

//! Unsigned 8-Bit value.
typedef unsigned char fcByte;

//! Unsigned 16-Bit value.
typedef unsigned short fcWord;

//! Unsigned 32-Bit value.
typedef unsigned int fcDword;

//! Unsigned 64-Bit value.
typedef unsigned long long fcQuad;

//! This enumeration defines the available FlexCard bus types.
//! \see #fcVersionCC
typedef enum fcBusType
{
	fcBusTypeFlexRay = 0,	//!< The FlexRay bus is selected.
	fcBusTypeCAN,			//!< The CAN bus is selected.
} fcBusType;

//! This enumeration defines the available FlexCard channels combination.
//! \see #fcMsgBufCfg
typedef enum fcChannel
{
    fcChannelNone = 0x00,                   //!< No FlexRay channel is selected.
    fcChannelA    = 0x01,                   //!< Only FlexRay channel A is selected.
    fcChannelB    = 0x02,                   //!< Only FlexRay channel B is selected.
    fcChannelBoth = fcChannelA | fcChannelB //!< FlexRay channel A + B are selected
} fcChannel;

//! This enumeration defines the available FlexCard communication controller index.
typedef enum fcCC
{
    fcCC1 = 0x00,	//!< The communication controller 1 is selected.
    fcCC2 = 0x01,	//!< The communication controller 2 is selected.
	fcCC3 = 0x02,	//!< The communication controller 3 is selected.
	fcCC4 = 0x03,	//!< The communication controller 4 is selected.
	fcCC5 = 0x04,	//!< The communication controller 5 is selected.
	fcCC6 = 0x05,	//!< The communication controller 6 is selected.
	fcCC7 = 0x06,	//!< The communication controller 7 is selected.
	fcCC8 = 0x07,	//!< The communication controller 8 is selected.
} fcCC;

//! This enumeration defines the possible communication controller POC states
//! (FlexRay Protocol Specification: vPOC!State). For more details about
//! communication controller POC states, please refer to FlexRay Protocol
//! Specification.
//! \see #fcbGetCcState, #fcbMonitoringStart
typedef enum fcState
{
    fcStateUnknown = 0,   //!< Communication controller state is not known
    fcStateConfig,        //!< Communication controller is in CONFIG state.
    fcStateNormalActive,  //!< Communication controller is in NORMAL_ACTIVE state.
    fcStateNormalPassive, //!< Communication controller is in NORMAL_PASSIVE state.
    fcStateHalt,          //!< Communication controller is in HALT state.
    fcStateReady,         //!< Communication controller is in READY state.
    fcStateStartup,       //!< Communication controller is in STARTUP state.
    fcStateWakeup,        //!< Communication controller is in WAKEUP state.
	fcStateMonitorMode,   //!< Communication controller is in MONITORMODE state.
} fcState;

//! This enumeration defines the CAN communication controller states. For more
//! details about CAN communication controller states, please refer to CAN
//! Protocol Specification.
//! \see #fcbCANGetCcState, #fcbCANMonitoringStart
typedef enum fcCANCcState
{
	fcCANCcStateUnknown = 0,	//!< Communication controller state is unknown.
	fcCANCcStateConfig,			//!< Communication controller is in configuration state.
	fcCANCcStateNormalActive,	//!< Communication controller is in normal active state.
	fcCANCcStateWarning,		//!< Communication controller is in error warning state.
								//!< At least one of the error counters has reached the
								//!< error warning limit of 96.
	fcCANCcStateErrorPassive,	//!< Communication controller is in Passive state. No
								//!< CAN messages can be sent anymore.
	fcCANCcStateBusOff,			//!< Communication controller is in bus off state.
} fcCANCcState;

//! This enumeration defines the possible communication controller wakeup states
//! (FlexRay Protocol Specification: vPOC!WakeupStatus). For more details about
//! communication controller wakeup states, please refer to the FlexRay Protocol
//! Specification.
typedef enum fcWakeupStatus
{
  fcWakeupStatusUndefined = 0,   //!< FlexRay Protocol Specification: UNDEFINED
  fcWakeupStatusReceiveHeader,   //!< FlexRay Protocol Specification: RECEIVE_HEADER
  fcWakeupStatusReceiveWUP,      //!< FlexRay Protocol Specification: RECEIVE_WUP
  fcWakeupStatusCollisionHeader, //!< FlexRay Protocol Specification: COLLISION_HEADER
  fcWakeupStatusCollisionWUP,    //!< FlexRay Protocol Specification: COLLISION_WUP
  fcWakeupStatusCollisionUnknown,//!< FlexRay Protocol Specification: COLLISION_UNKNOWN
  fcWakeupStatusTransmitted,     //!< FlexRay Protocol Specification: TRANSMITTED
} fcWakeupStatus;

//! This enumeration defines the different states of the FlexRay transceivers.
typedef enum fcTransceiverState
{
	//! Transceiver is in normal mode and is able to transmit and receive data
	//! via the FlexRay bus.
	fcTransceiverNormal,

	//! Transceiver is in low power mode and is not able to transmit and receive
	//! data, but is able to detect wake-up events on the bus. If a wakeup event
	//! is detected the event fcNotificationTypeFRWakeup is fired.
	fcTransceiverSleep,
} fcTransceiverState;

//! This enumeration defines the supported communication symbols when the
//! communication controller is in POC state NORMAL_ACTIVE. For more details
//! about these symbols, please refer to the FlexRay Protocol Specification. To
//! send a wakeup symbol (WUS) or collision avoidance symbol (CAS), refer to the
//! function #fcbMonitoringStart.
typedef enum fcSymbolType
{
   fcMediaAccessTestSymbolType = 1, //!< Media Access Test Symbol (MTS)
} fcSymbolType;

//! This enumeration defines the communication controller types supported by the
//! API. The parameter CCType of the structure #fcVersionCC, which is returned by
//! the function #fcbGetEnumFlexCardsV3, indicates the communication controller
//! used by the FlexCard.
//! \see #fcbGetEnumFlexCardsV3, #fcInfoHw
typedef enum fcCCType
{
    Undefined = 0,	//!< Undefined communication controller
    FreeScale_FPGA,	//!< Communication controller is a FreeScale FPGA
    Bosch_E_Ray,	//!< Communication controller is a Bosch E-Ray
	Bosch_DCAN,		//!< Communication controller is a Bosch DCAN
} fcCCType;

//! This structure describes the version of a FlexCard component (hardware or
//! software). The function #fcbGetEnumFlexCardsV3 returns the version numbers
//! for all FlexCard components.
//! \see #fcbGetEnumFlexCardsV3, #fcInfoHw, #fcInfoSw
typedef struct fcVersionNumber
{
    fcDword Major;   //!< An increment indicates a modification which is not
                     //!< downwardly compatible.
    fcDword Minor;   //!< An increment indicates a light-weight modification
    fcDword Update;  //!< Indicates an update (bug fix) for a minor version
    fcDword Release; //!< 0 indicates a release version, greater than 0
                     //!< indicates a test version
} fcVersionNumber;

//! This enumeration defines the different device ids available
//! \see #fcInfoHw
typedef enum fcFlexCardDeviceId
{
	fcNoDevice				= 0,	//!< No FlexCard was detected.

	fcFlexCardCycloneII		= 5,	//!< A 'FlexCard Cyclone II' was detected
	fcFlexCardCycloneIIPXI	= 6,	//!< A 'FlexCard PXI' was detected
	fcFlexCardPMC			= 7,	//!< A 'FlexCard PMC' was detected
	fcFlexCardCycloneIISE	= 8,	//!< A 'FlexCard Cyclone II SE' was detected
	fcFlexCardPMCII			= 9,	//!< A 'FlexCard PMC II' was detected
} fcFlexCardDeviceId;

//! This structure provides information about the useable number of
//! communication controllers of the FlexCard hardware.
//! \see #fcInfoHw, #fcbGetNumberCCs, #fcbGetInfoFlexCard
typedef struct fcNumberCC
{
	fcByte FlexRay;			//!< Number of FlexRay communication controllers.
	fcByte CAN;				//!< Number of CAN communication controllers.
	fcByte LIN;				//!< Number of LIN communication controllers.
	fcByte MOST;			//!< Number of MOST communication controllers.
	fcByte Reserved[4];		//!< Reserved for future use.
} fcNumberCC;

//! This structure provides version information about the available FlexCard
//! communication controllers.
//! \see #fcInfoHw
typedef struct fcVersionCC
{
	fcBusType BusType;					//!< Communication controller bus type
	fcCC CCIndex;						//!< Communication controller identifier
	fcCCType CCType;					//!< Communication controller type
	fcVersionNumber CCVersion;			//!< Communication controller version
	fcVersionNumber Protocol;			//!< Protocol version of the bus type
	struct fcVersionCC* pNext;			//!< Pointer to the next cc version. If the pointer
										//!< is NULL, there are no more cc versions available.
	fcDword IncorrectPhysicalLayer : 1;	//!< Physical layer module detection. A value <> 0
										//!< indicates a mismatch between communication
										//!< controller type and physical layer module.
	fcDword Reserved[2];				//!< Reserved for future use
} fcVersionCC;

//! THIS STRUCTURE IS OBSOLETE  (Please use #fcInfoHwSw now.)
//! This structure provides version information about the FlexCard hardware and
//! software components.
//! \see #fcbGetEnumFlexCards, #fcbGetEnumFlexCardsV2, #fcInfo, #fcInfoV2
typedef struct fcVersion
{
	fcVersionNumber BaseDll;		//!< DLL Base Version
	fcVersionNumber DeviceDriver;	//!< Device driver version
	fcVersionNumber Firmware;		//!< Firmware (gateway software) version
	fcVersionNumber Hardware;		//!< FlexCard hardware version
	fcCCType CCType;				//!< Communication controller type
	fcVersionNumber CC;				//!< Communication controller version
	fcVersionNumber BusGuardian;	//!< Bus Guardian version
	fcVersionNumber Protocol;		//!< FlexRay Protocol version
	fcDword Serial;					//!< FlexCard serial number. A zero value indicates
									//!< a non-valid FlexCard serial number.
	fcFlexCardDeviceId DeviceId;	//!< FlexCard Device ID
	fcVersionCC* pVersionCC;		//!< Pointer to version information about the
									//!< available communication controllers.
	fcDword Reserved[2];			//!< Reserved for internal purpose
} fcVersion;

//! THIS STRUCTURE IS OBSOLETE  (Please use #fcInfoHwSw now.)
//! This structure provides information about the components and the identifier
//! of a FlexCard. If more than one FlexCard is detected on the system, the
//! #fcbGetEnumFlexCards function returns a linked list of this structure. If a
//! connection to a FlexCard is already opened, this FlexCard does not appear in
//! this list.
//! \see #fcbGetEnumFlexCards, #fcVersion
typedef struct fcInfo
{
	fcDword FlexCardId;		//!< Unique number used to identify a FlexCard. This id
							//!< is required to open a connection to the FlexCard.
	fcVersion Version;		//!< Version information about hardware and software
							//!< components of the FlexCard.
	struct fcInfo* pNext;	//!< Pointer to the next available FlexCard, if no other
							//!< FlexCard exists, pNext is a null pointer.
} fcInfo;

//! THIS STRUCTURE IS OBSOLETE  (Please use #fcInfoHwSw now.)
//! This structure provides information about the components, the identifier and the
//! current device state of a FlexCard. If more than one FlexCard is detected on the
//! system, the #fcbGetEnumFlexCardsV2 function returns a linked list of this structure.
//! \see #fcbGetEnumFlexCardsV2, #fcVersion
typedef struct fcInfoV2
{
	fcDword FlexCardId;		//!< Unique number used to identify a FlexCard. This id
							//!< is required to open a connection to the FlexCard.
	fcVersion Version;		//!< Version information about hardware and software
							//!< components of the FlexCard.
	fcDword Busy;			//!< The current device state. A value <> 0 indicates
							//!< a connection to this FlexCard is already opened.
	fcDword Reserved;		//!< Reserved for future use
	struct fcInfoV2* pNext;	//!< Pointer to the next available FlexCard, if no other
							//!< FlexCard exists, pNext is a null pointer.
} fcInfoV2;

//! This structure provides information about the software components of a FlexCard.
//! \see #fcbGetEnumFlexCardsV3, #fcbGetInfoFlexCard, #fcInfoHwSw, #fcInfoHw
typedef struct fcInfoSw
{
	fcVersionNumber VersionBaseDll;			//!< DLL Base Version
	fcVersionNumber VersionDeviceDriver;	//!< Device driver version
	fcDword LicensedForLinuxDriver : 1;		//!< Valid license for FlexCard Linux driver
	fcDword LicensedForWindowsDriver : 1;	//!< Valid license for FlexCard Windows driver
	fcDword LicensedForXenomaiDriver : 1;	//!< Valid license for FlexCard Xenomai driver
	fcDword Reserved[4];					//!< Reserved for future use
} fcInfoSw;

//! This structure provides information about the hardware components of a FlexCard.
//! \see #fcbGetEnumFlexCardsV3, #fcbGetInfoFlexCard, #fcInfoHwSw, #fcInfoSw
typedef struct fcInfoHw
{
	fcQuad Serial;						//!< FlexCard serial number. A zero value indicates
										//!< a non-valid FlexCard serial number.
	fcFlexCardDeviceId DeviceId;		//!< FlexCard Device ID
	fcVersionNumber VersionFirmware;	//!< Firmware (gateway software) version
	fcVersionNumber VersionHardware;	//!< FlexCard hardware version
	fcNumberCC SupportedCCs;			//!< Possible FlexCard communication controller
										//!< counts with the hardware.
	fcNumberCC LicensedCCs;				//!< Licensed FlexCard communication controller
										//!< counts with the hardware.
	fcNumberCC UseableCCs;				//!< Useable FlexCard communication controller
										//!< counts for the software.
	fcVersionCC* pVersionCC;			//!< Pointer to version information about the
										//!< useable communication controllers.
	fcDword Reserved[8];				//!< Reserved for future use
} fcInfoHw;

//! This structure provides information about the components, the identifiers and the
//! current device state of a FlexCard. If more than one FlexCard is detected on the
//! system, the #fcbGetEnumFlexCardsV3 function returns a linked list of this structure;
//! the function #fcbGetInfoFlexCard function returns an item of this structure.
//! \see #fcbGetEnumFlexCardsV3, #fcbGetInfoFlexCard, #fcInfoHw, #fcInfoSw
typedef struct fcInfoHwSw
{
	fcDword FlexCardId;			//!< Unique number used to identify a FlexCard. This id
								//!< is required to open a connection to the FlexCard.
	fcDword UserDefinedCardId;	//!< User defined number used to identify a FlexCard. This
								//!< id is not unique! A zero value indicates a non-valid
								//!< or non-exist identifier.
	fcInfoSw InfoSoftware;		//!< Information about software components of the FlexCard.
	fcInfoHw InfoHardware;		//!< Information about hardware components of the FlexCard.
	fcDword Busy : 1;			//!< The current device state. A value <> 0 indicates
								//!< a connection to this FlexCard is already opened.
	struct fcInfoHwSw* pNext;	//!< Pointer to the next available FlexCard. If no other
								//!< FlexCard exists, pNext is a null pointer.
	fcDword Reserved[2];		//!< Reserved for future use
} fcInfoHwSw;

//! This enumeration defines the different modes available, used to monitor a
//! FlexRay cluster.
//! \see #fcbMonitoringStart
typedef enum fcMonitoringModes
{
    //! Normal operation mode of the FlexCard.
	//! First, the FlexCard try to synchronize itself with the cluster. Once the
	//! synchronization succeeds, the FlexCard enters in the NORMAL_ACTIVE state
	//! and is able to transmit and receive FlexRay frames, symbols and errors,
	//! as previously configured.
	fcMonitoringNormal,

  	//! Debug operation mode of the FlexCard using the E-Ray communication controller.
	//! The FlexCard does not try to synchronize itself with the cluster and is
	//! only able to receive FlexRay frames, symbols and errors from the FlexRay
	//! bus. This mode does not allow transmission; it is therefore not possible
	//! to perform a start-up or a wakeup. This mode is adapted for debugging
	//! purpose (e.g. start up of a FlexRay network fails).
	//! \note This mode is an E-Ray specific feature and is thereby only
	//! 	  available for the E-Ray FlexCard version. To receive frames with
	//! 	  this mode, you have to configure receive buffers. The FIFO receive
	//! 	  buffers aren't working in this mode.
    fcMonitoringDebug,

	//! Special debug operation mode of the FlexCard which allows the reception of
	//! all frames without any message buffer and controller configuration. The
	//! only parameter to be set is the baud rate (Register 0x0090: 10
	//! MBit/s: 0x00000000, 5 MBit/s: 0x00004000, 2.5 MBit/s: 0x00008000). This
	//! mode does not allow transmission; it is therefore not possible to
	//! perform a start-up or a wakeup. This mode is adapted for debugging
	//! purpose (e.g. start up of a FlexRay network fails or to monitor an unknown
	//! network). The timestamp accuracy in this mode is +/-2 us.
	//! Incorrect data will be interpreted as received FlexRay frames. In this case
	//! the Valid Frame Bit is not set.
	fcMonitoringDebugAsynchron,

	//! This mode combines the mode fcMonitoringDebugFull and
	//! fcMonitoringNormal. The mode fcMonitoringDebugFull is used to receive
	//! all frames during startup. Unlike the fcMonitoringDebugFull this mode
	//! allows the transmission of sync frames. After the startup completed
	//! successfully, the FlexCard switches directly to the mode fcMonitoringNormal.
	fcMonitoringDebugAsynchronBeforeStartup,
} fcMonitoringModes;

//-----------------------------------------------------------------------------
//          Configuration
//-----------------------------------------------------------------------------

//! For the transmission and reception of FlexRay frames the communication
//! controller provides different types of message buffers. Each message buffer
//! can be assigned with one of the following specific types.
//! \note In certain cases, it is not possible to receive all frames with only
//! 	  receive message buffers. To ensure that all frames will be received,
//! 	  we recommend to configure some FIFO message buffers.
//! \see #fcMsgBufCfg, #fcMsgBufCfgFifo, #fcMsgBufCfgRx, fcMsgBufCfgTx
typedef enum fcMsgBufType
{
    //! The message buffer is not used
    fcMsgBufNone,

    //! The message buffer is used as a receive buffer (e.g. to analyse a
	//! specific frame).
    fcMsgBufRx,

    //! The message buffer is used as a transmit buffer (e.g. to transmit a
	//! message on a specific communication slot).
    fcMsgBufTx,

    //! The message buffer is used as a receive FIFO buffer. In that case, it
	//! will receive frames from different communication slots..
    fcMsgBufFifo,
} fcMsgBufType;

//! This enumeration defines the different modes of transmission.
//! \see #fcMsgBufCfgTx
typedef enum fcMsgBufTxMode
{
    fcMsgBufTxSingleShot,   //!< Frame is transmitted once only if its
							//!< corresponding message buffer content has been
							//!< actualized and both frame id and cycle filter
							//!< are matching. The function #fcbTransmit
							//!< actualize the content of a given message
							//!< buffer.
    fcMsgBufTxContinous,    //!< Frame is transmitted each time when both the
							//!< frame id and cycle filter are matching,
							//!< regardless if its corresponding message buffer
							//!< content has been actualized or not.
} fcMsgBufTxMode;

//! Maximum number of 2-byte payload data words
//! \see #fcMsgBufCfgFifo, #fcMsgBufCfgRx, fcMsgBufCfgTx
const fcByte fcPayloadMaximum = 127;

//! This structure specifies the configuration of a FIFO buffer. The FIFO
//! message buffers are used to receive FlexRay frames from different
//! communication slots and allow therefore to receive more frames than message
//! buffers exist.
//!
//! \note Modifying the FIFO configuration may affect the ability to receive all
//! 	  frames (e.g. setting the RejectInStaticSegment flag to 1 will unable
//! 	  the FlexCard to monitor frames in the static segment).Configuring
//! 	  (fcbConfigureMessageBuffer) the FIFO is only possible when the
//! 	  communication controller is in its configuration state, fcStateConfig.
//! 	  A reconfiguration (fcbReconfigureMessageBuffer) is not allowed for
//! 	  this buffer type.
typedef struct fcMsgBufCfgFifo
{
    //! Defines the acceptance filter used for frame id rejection. A zero value
	//! means that no frame is rejected.
    fcDword FrameIdFilter : 11;

    //! Specifies the relevant bits used for rejection filtering.
    fcDword RejectionMask : 11;

    //! Defines the maximum number of 2-byte payload words received.
    fcDword PayloadLengthConfigured : 7;

    //! Set this flag to 1 to reject all received static frames of the FIFO. A
	//! zero value deactivates the FIFO static segment rejection. This feature
	//! is not available for FlexCard Cyclone I Card.
    fcDword RejectInStaticSegment : 1;

    //! Set this flag to 1 to reject all received null frames of the FIFO. A
	//! zero value deactivates the FIFO null frame rejection. This feature is
	//! not available for FlexCard Cyclone I Card.
    fcDword RejectNullFrames : 1;

    //!< Reserved for future use
    fcDword Reserved;
} fcMsgBufCfgFifo;

//! This structure specifies the configuration of a receive message buffer. This
//! buffer type should be used to analyze a specific communication slot (=frame
//! id). Only buffers with an Id between 2 and 34 can be used as receive message
//! buffer.
//!
//! \note FlexCards with FreeScale communication controllers can only receive
//! 	  frames through FIFO message buffers (receive message buffers are not
//! 	  available). For FlexCard with the E-Ray communication controller the
//! 	  flag RxAcknowledgeEnable have to be set to receive frames with the
//! 	  function fcbReceive.
typedef struct fcMsgBufCfgRx
{
    //! Defines the slot (=frame id) to be received in this message buffer. With
	//! the function fcbReconfigureMessageBuffer, this parameter can be changed
	//! while monitoring is active.
    fcDword FrameId : 11;

    //! Defines the number of 2-byte payload words to be received. This
	//! parameter can be changed while monitoring is active. To this end, use
	//! the function #fcbReconfigureMessageBuffer and set this parameter with a
	//! value between 0 and PayloadLengthMax. The reconfiguration of this
	//! parameter for message buffers assigned to the static segment is not
	//! allowed.
    fcDword PayloadLengthConfigured : 7;

    //! Defines the maximum payload reserved for this buffer in the message ram.
	//! This E-Ray specific parameter sets the range for the payload
	//! reconfiguration. This parameter can not be changed while monitoring is
	//! active.
    fcDword PayloadLengthMax : 7;

    //! Enables message buffer interrupt. This flag must be set to 1 to allow
	//! the function fcbReceive to get the received frame. This parameter can be
	//! changed while monitoring is active. To this end, use the function
	//! #fcbReconfigureMessageBuffer.
    fcDword RxAcknowledgeEnable : 1;

    //! Reserved for future use
    fcDword Reserved;
} fcMsgBufCfgRx;

//! This structure specifies the configuration of a transmit message buffer.
//! This buffer type is used to transmit a frame on a specific communication
//! slot.
typedef struct fcMsgBufCfgTx
{
    //! Defines the slot (=frame id) assigned to the transmit message buffer.
	//! With the function #fcbReconfigureMessageBuffer, this parameter can be
	//! changed while monitoring is active.
    fcDword FrameId : 11;

    //! Defines the number of 2-byte payload words to be transmitted. This
	//! parameter can be changed while monitoring is active. To this end, use
	//! the function #fcbReconfigureMessageBuffer and set this parameter with a
	//! value between 0 and PayloadLengthMax. The reconfiguration of this
	//! parameter for message buffers assigned to the static segment is not
	//! allowed.
    fcDword PayloadLength : 7;

    //! Defines the maximum payload reserved for this buffer in the message ram.
	//! This E-Ray specific parameter sets the range for the payload
	//! reconfiguration. This parameter can not be changed while monitoring is
	//! active.
    fcDword PayloadLengthMax : 7;

    //! This parameter is protocol specific. For more information, refer to
	//! FlexRay Protocol Specification. With the function
	//! fcbReconfigureMessageBuffer, this parameter can be changed while
	//! monitoring is active.
    fcDword PayloadPreambleIndicator : 1;

    //! Set this flag to 1 to indicate that the frame is a sync frame. This
	//! parameter can not be changed while monitoring is active.
    fcDword SyncFrameIndicator : 1;

    //! Set this flag to 1 to indicate that the frame is a start-up frame. This
	//! parameter can not be changed while monitoring is active.
    fcDword StartupFrameIndicator : 1;

	//! Enables the user to become an acknowledge packet (fcTxAcknowledgePacket)
	//! once a frame is transmitted (includes null frames). With the function
	//! #fcbReconfigureMessageBuffer, this parameter can be changed while
	//! monitoring is active. This feature is only available on FlexCard based
	//! on E-Ray communication controller.
    fcDword TxAcknowledgeEnable : 1;

     //! Type of transmission (refer to fcMsgBufTxMode). With the function
	//! fcbReconfigureMessageBuffer, this parameter can be changed while
	//! monitoring is active.
    fcMsgBufTxMode TransmissionMode;

	//! Set this flag to 1 to get TxAchnowledge packet for transmitted null
	//! frames. This flag is only evaluated if the TxAcknowledgeEnable flag is
	//! set.
	fcDword TxAcknowledgeShowNullFrames : 1;

	//! Set this flag to 1 to get the payload of the transmitted frame. The payload length
	//! of generated TxAchnowledge packet will otherwise be set to zero. This flag is only
	//! evaluated if the TxFrameEnable flag is set.
	fcDword TxAcknowledgeShowPayload : 1;

	//! Reserved for future use
	fcDword Reserved:29;
} fcMsgBufCfgTx;

//! This structure describes the configuration of a message buffer.
//! \see #fcbConfigureMessageBuffer, #fcbReconfigureMessageBuffer,
//!      #fcbGetCcMessageBuffer
typedef struct fcMsgBufCfg
{
    //! Defines the buffer type (FIFO, receive or transmit buffer)
    fcMsgBufType Type;

    //! Defines the channel(s) assigned to this buffer. With the function
	//! fcbReconfigureMessageBuffer, this parameter can only be changed while
	//! monitoring is active for receive and transmit buffer. For the
	//! configuration of a transmit or a receive message buffer assigned to a
	//! dynamic frame, only fcChannelA or fcChannelB is allowed.
    fcChannel ChannelFilter;

    //! Defines the filter used by the message buffer for cycle counter
	//! filtering. A zero value means that no cycle counter filtering is used.
	//! The cycle counter filter is composed of two parameters. The first one
	//! determines the cycle repetition and the second one the offset (the first
	//! cycle). The cycle repetition must be given in the form of 2^x where x is
	//! a number between 0 and 7. The offset must be less than the cycle
	//! repetition value. With the function fcbReconfigureMessageBuffer, this
	//! parameter can only be changed while monitoring is active for receive and
	//! transmit buffer.
    fcDword CycleCounterFilter : 7;

    union
    {
        fcMsgBufCfgFifo Fifo;
        fcMsgBufCfgRx   Rx;
        fcMsgBufCfgTx   Tx;
    };

    fcDword Reserved[2]; //!< Reserved for future use
} fcMsgBufCfg;

//! This structure describes the configuration of a FlexRay CC Timer0.
//! \see #fcbSetCcTimerConfig, #fcbGetCcTimerConfig, #fcbCalculateMacrotickOffset
typedef struct fcCcTimerCfg
{
	//! Defines the cc timer mode. Set to true for continuous mode or false
	//! for single-shot mode.
	fcDword ContinuousMode : 1;

	//! Defines the filter used by the cc timer for cycle counter filtering.
	//! A zero value means that no cycle counter filtering is used. The cycle
	//! counter filter is composed of two parameters. The first one determines the
	//! cycle repetition and the second one the offset (the first cycle). The cycle
	//! repetition must be given in the form of 2^x where x is a number between 0
	//! and 7. The offset must be less than the cycle repetition value.
	fcDword CycleCounterFilter : 7;

	//! Defines the macrotick offset from the beginning of the cycle where the
	//! cc timer interrupt is to occur. The cc timer interrupt occurs at this
	//! offset for each cycle of the cycle counter filter.
	fcDword MacrotickOffset : 14;
} fcCcTimerCfg;

//! This enumeration defines various positions in a FlexRay cycle.
//! \see #fcbCalculateMacrotickOffset
typedef enum fcCyclePos
{
	//! No cycle position defined
	fcCyclePosNotDefined = 0,

	//! Defines the start of a static slot
	fcCyclePosStaticSlot,
	//! Defines the start of a dynamic mini slot
	fcCyclePosDynamicMiniSlot,

	//! Defines the end of the static segment
	fcCyclePosEndStaticSegment,
	//! Defines the start of the dynamic segment
	fcCyclePosStartDynamicSegment,
	//! Defines the end of the dynamic segment
	fcCyclePosEndDynamicSegment,
	//! Defines the start of the symbol window
	fcCyclePosStartSymbolWindow,
	//! Defines the end of the symbol window
	fcCyclePosEndSymbolWindow,
	//! Defines the start of the network idle time
	fcCyclePosStartNetworkIdleTime,
} fcCyclePos;

//! THIS ENUMARATION IS OBSOLETE  (Please use #fcTriggerConditionEx now.)
//! This enumeration defines the conditions available for a trigger configuration.
typedef enum fcTriggerCondition
{
    //! Passive mode conditions: input trigger is detected on falling edge
    fcTriggerConditionFallingEdge    = 1,
    //! Passive mode conditions: input trigger is detected on rising edge
    fcTriggerConditionRisingEdge     = 2,

    //! Active mode conditions: output trigger is set on start of a new FlexRay
    //! cycle
    fcTriggerConditionCycleStart     = 3,
    //! Active mode conditions: output trigger is set by the user
    fcTriggerConditionUser           = 4,
    //! Active mode conditions: output trigger is set if an error was detected
    fcTriggerConditionErrorDetected  = 5,
    //! Active mode conditions: output trigger is set when the startup was
    //! completed
    fcTriggerConditionStartupCompleted = 6,

    //! Timer mode conditions:  Internal trigger is set by the software timer
    //! (neither input nor output trigger signal is used)
    fcTriggerConditionTimer          = 7,
} fcTriggerCondition;

//! THIS ENUMARATION IS OBSOLETE  (Please use #fcTriggerConditionEx now.)
//! This enumeration defines the different trigger types.
//! \see #fcTriggerCfg
typedef enum fcTriggerType
{
    fcTriggerTypeHardware   = 1, //!< Hardware trigger
    fcTriggerTypeSoftware   = 2, //!< Software trigger
} fcTriggerType;

//! THIS ENUMARATION IS OBSOLETE  (Please use #fcTriggerConditionEx now.)
//! This enumeration defines the different trigger modes.
//! \see #fcTriggerCfgHardware, #fcTriggerCfgSoftware
typedef enum fcTriggerMode
{
    fcTriggerModeActive  = 1,   //!< Active mode: triggered by FlexCard or by user
    fcTriggerModePassive = 2,   //!< Passive mode: triggered by external hardware
    fcTriggerModeTimer   = 3,   //!< Timer mode: triggered by software timer.
} fcTriggerMode;

//! THIS ENUMARATION IS OBSOLETE  (Please use #fcTriggerConfigurationEx now.)
//! This structure configures the hardware trigger. In the passive mode, the
//! FlexCard waits for trigger events on its input line and generates an
//! #fcTriggerInfoPacket object each time a trigger event is received. In this
//! mode, the parameter Condition specifies on which condition the input signal
//! will be recognized as a trigger event. In the active mode, the FlexCard
//! generates a pulse on its output line when a trigger event is signaled. In
//! this mode, the parameter Condition specifies on which condition a pulse will
//! be generated by the FlexCard. For information about the pin assignment of
//! the input and output line, refer to the user manual of the FlexCard.
//! \see #fcTriggerCfg, #fcTriggerCfgSoftware
typedef struct fcTriggerCfgHardware
{
    //! Set the trigger mode (active or passive mode). The hardware trigger does
	//! not support the timer mode.
    fcTriggerMode Mode;

    //! Depending on the mode the following conditions can be used: Passive
    //! mode: - Falling edge (Trigger packet is generated on falling
    //!               edge of the input signal)
    //!               - Rising edge (Trigger packet is generated on rising edge
    //!               of the input signal)
    //! Active mode : - Cycle start (A pulse is generated on the output line
	//!                 when a new cycle starts)
    //!               - User (A pulse is generated on the output line when the
	//! 				user is calling the function fcbTrigger)
    //!               - Error (A pulse is generated on the output line when an
	//! 				error occurred)
    //!               - Start up completed (A pulse is generated on the output
	//! 				line when the start up was completed)
    fcTriggerCondition Condition;
} fcTriggerCfgHardware;

//! THIS STRUCTURE IS OBSOLETE  (Please use #fcTriggerConfigurationEx now.)
//! This structure configures the software trigger. In active mode an
//! #fcTriggerInfoPacket object is generated each time the function #fcbTrigger is
//! called. In the timer mode an #fcTriggerInfoPacket object is generated every
//! TimePeriod millisecond. A zero TimePeriod means that no #fcTriggerInfoPacket
//! will be generated.
//! \see #fcTriggerCfg, #fcTriggerCfgHardware
typedef struct fcTriggerCfgSoftware
{
    //! Set the trigger mode (active or timer mode). The software trigger does
	//! not support the passive mode.
    fcTriggerMode Mode;

    //! This parameter is only used in timer mode. Every TimePeriod milliseconds
	//! (range: 0 - 400000) a trigger packet will be generated
    fcDword  TimePeriod;
} fcTriggerCfgSoftware;

//! THIS STRUCTURE IS OBSOLETE  (Please use #fcTriggerConfigurationEx now.)
//! This structure is used for the configuration of a trigger. Only one trigger
//! at a time (hardware or software) can be used and the conditions cannot be
//! combined.
//! \see #fcbTrigger
typedef struct fcTriggerCfg
{
    //! Type of trigger (hardware or software)
    fcTriggerType Type;
    union
    {
        fcTriggerCfgHardware Hardware;   //!< Configuration of hardware
                                         //!< trigger
        fcTriggerCfgSoftware Software;   //!< Configuration of software
                                         //!< trigger
    };
} fcTriggerCfg;

//! This enumeration defines the conditions available for the trigger configuration of FlexCard.
typedef enum fcTriggerConditionEx
{
	fcTriggerIn							= 0x00000002,	//!IN:  A trigger packet is generated as soon as the set edge (falling/rising) is detected on the IN trigger line
	fcTriggerOutOnPulse					= 0x00000004,	//!OUT: A signal is generated on the OUT trigger line as soon as the condition is set to the driver
	fcTriggerInOnSWPulse				= 0x00000008,	//!IN:  A trigger packet is generated as soon as the condition is set to the driver
	fcTriggerInOnSWTimer				= 0x00000010,	//!IN:  A trigger packet is generated by a set time interval
	fcTriggerOutOnCycle					= 0x00000040,	//!OUT: A signal is generated on the OUT trigger line at each start of a set FlexRay cycle
	fcTriggerOutOnSlotChA				= 0x00000080,	//!OUT: A signal is generated on the OUT trigger line at each start of a set slot on channel A
	fcTriggerOutOnSlotChB				= 0x00000100,	//!OUT: A signal is generated on the OUT trigger line at each start of a set slot on channel B
	fcTriggerOutOnSlotInCycleChA		= 0x00000200,	//!OUT: A signal is generated on the OUT trigger line at each start of a set slot in a set cycle on channel A
	fcTriggerOutOnSlotInCycleChB		= 0x00000400,	//!OUT: A signal is generated on the OUT trigger line at each start of a set slot in a set cycle on channel B
	fcTriggerOutOnCycleStart			= 0x00010000,	//!OUT: A signal is generated on the OUT trigger line at a cycle start
	fcTriggerOutOnErrorDetected			= 0x00020000,	//!OUT: A signal is generated on the OUT trigger line at detected error
	fcTriggerOutOnStartupCompleted		= 0x00040000,	//!OUT: A signal is generated on the OUT trigger line at a 'startup completed'
	fcTriggerOutOnStartDynamicSegment	= 0x00080000,	//!OUT: A signal is generated on the OUT trigger line at SDS  (start of dynamic segment)
	fcTriggerOutOnErrorSFBM				= 0x00100000,	//!OUT: A signal is generated on the OUT trigger line at error SFBM (sync frame below minimum)
	fcTriggerOutOnErrorSFO				= 0x00200000,	//!OUT: A signal is generated on the OUT trigger line at error SFO	(sync frame overflow)
	fcTriggerOutOnErrorCCF				= 0x00400000,	//!OUT: A signal is generated on the OUT trigger line at error CCF  (clock correction failure)
	fcTriggerOutOnErrorSBVA				= 0x00800000,	//!OUT: A signal is generated on the OUT trigger line at error SBVB (slot boundary violation channel A)
	fcTriggerOutOnErrorPERR				= 0x01000000,	//!OUT: A signal is generated on the OUT trigger line at error PERR (parity error)
	fcTriggerOutOnErrorEDA				= 0x02000000,	//!OUT: A signal is generated on the OUT trigger line at error EDA  (error detected on channel A)
	fcTriggerOutOnErrorLTVA				= 0x04000000,	//!OUT: A signal is generated on the OUT trigger line at error LTVA (latest transmit violation channel A)
	fcTriggerOutOnErrorTABA				= 0x08000000,	//!OUT: A signal is generated on the OUT trigger line at error TABA (transmission across boundary channel A)
	fcTriggerOutOnErrorEDB				= 0x10000000,	//!OUT: A signal is generated on the OUT trigger line at error EDB  (error detected on channel B)
	fcTriggerOutOnErrorLTVB				= 0x20000000,	//!OUT: A signal is generated on the OUT trigger line at error LTVB (latest transmit violation channel B)
	fcTriggerOutOnErrorTABB				= 0x40000000,	//!OUT: A signal is generated on the OUT trigger line at error TABB (transmission across boundary channel B)
	fcTriggerOutOnErrorSBVB				= 0x80000000,	//!OUT: A signal is generated on the OUT trigger line at error SBVB (slot boundary violation channel B)
	fcTriggerOutOnErrorAll				= 0xFFF00000,	//!OUT: A signal is generated on the OUT trigger line at detected an error
} fcTriggerConditionEx;

//! This structure is used for the configuration of a trigger. The trigger
//! conditions can be combined for Cyclone II (SE) devices. For PMC/PCI or PMC II
//! devices no conditions can be combined. To reset all conditions the
//! parameter conditions should be set to 0.
//! The condition fcTriggerIn or fcTriggerPMCIn demands to set the parameter onEdge.
//! The condition fcTriggerInOnSWTimer demands to set the parameter onTimePeriod.
//! The condition fcTriggerOutOnCycle demands to set the parameter onCycle.
//! The condition fcTriggerOutOnSlotChA demands to set the parameter onSlotChA.
//! The condition fcTriggerOutOnSlotChB demands to set the parameter onSlotChB.
//! The condition fcTriggerOutOnSlotInCycleChA demands to set the parameter onSlotChA and onCycle.
//! The condition fcTriggerOutOnSlotInCycleChB demands to set the parameter onSlotChB and onCycle.
//! The condition fcTriggerPMCOutOnErrorDetected demands to set the parameter onSlotChB and onCycle TriggerGeneratingCC.
//! The condition fcTriggerPMCOutOnStartupCompleted demands to set the parameter onSlotChB and onCycle TriggerGeneratingCC.
//! The condition fcTriggerPMCOutOnCycleStart demands to set the parameter onSlotChB and onCycle TriggerGeneratingCC.
//! \see #fcbSetTrigger #fcTriggerConditionEx
typedef struct fcTriggerConfigurationEx
{
	//! This Parameter can either be set to 0 to reset all triggers or can be set
	//! to an OR-ed combination of conditions available in fcTriggerConditionEx for
	//! Cyclone II (SE) devices or can be set to a condition available in
	//! fcTriggerConditionPMC for PMC/PCI and PMC II devices.
    fcDword Condition;

	//! This parameter has to be set when the condition fcTriggerIn or fcTriggerPMCIn is chosen.
	//! Valid values are 0 = falling edge and 1 = rising edge.
	fcDword onEdge;

	//! for Cyclone II (SE) only:
	//! -------------------------
	//! This parameter has to be set when at least on of the conditions fcTriggerOutOnCycle,
	//! fcTriggerOutOnSlotInCycleChA and fcTriggerOutOnSlotInCycleChB are chosen.
	//! Valid values range from 0 to 63.
	fcDword onCycle;
	//! This parameter has to be set when at least on of the conditions fcTriggerOutOnSlotChA
	//! or fcTriggerOutOnSlotInCycleChA are chosen.
	//! Valid values range from 1 to 2047.
	fcDword onSlotChA;
	//! This parameter has to be set when at least on of the conditions fcTriggerOutOnSlotChB
	//! or fcTriggerOutOnSlotInCycleChB are chosen.
	//! Valid values range from 1 to 2047.
	fcDword onSlotChB;
	//! This parameter is only used in timer mode. Every TimePeriod milliseconds
	//! (range: 0 - 400000) a trigger packet will be generated
	fcDword onTimePeriod;

	fcDword Reserved1[4];

	//! for PMC/PCI or PMC II only:
	//! ---------------------------
	//! This parameter sets the trigger line which should be configured.
	//! (On PMC cards every trigger line can be used as an IN trigger or OUT trigger line.)
	//! Valid values range from 1 to 2.
	fcDword TriggerLineToConfigure;
	//! This parameter has to be set when CC dependent trigger conditions were
	//! set on a FlexCard PMC (II). Valid values range from fcCC1 to fcCC4.
	fcCC TriggerGeneratingCC;

	fcDword Reserved2[4];
} fcTriggerConfigurationEx;

//-----------------------------------------------------------------------------
//          Events
//-----------------------------------------------------------------------------

//! This enumeration defines different notification types. These types are used
//! in the function #fcbSetEventHandleV2, #fcbSetEventHandleSemaphore or fcbWaitForEventV2
//! to specify on which kind of event the application has to be notified.
//! \see #fcbFRMonitoringStart, #fcbSetTimer, #fcbSetEventHandleV2,
//!		 #fcbSetEventHandleSemaphore or fcbWaitForEventV2
typedef enum fcNotificationType
{
	//! Used to notify that a new cycle has started and that probably new data has been received.
	fcNotificationTypeCycleStarted = 1,
	fcNotificationTypeFRCycleStarted = fcNotificationTypeCycleStarted,

	//! Used to notify that the timer interval has elapsed. This notification
	//! requires the internal timer of the FlexCard to be enabled (See fcbSetTimer).
	fcNotificationTypeTimer = 2,

	//! Used to notify that one of the transceivers has received a wakeup event
	//! (only if one of the transceivers was in sleep mode).
	fcNotificationTypeWakeup = 3,
	fcNotificationTypeFRWakeup = fcNotificationTypeWakeup,

	//! Reserved 4 - 11

	//! Used to notify that the configured cc timer macrotick offset has elapsed.
	fcNotificationTypeCcTimer = 12,
	fcNotificationTypeFRCcTimer = fcNotificationTypeCcTimer,

	//! For internal use only.
	fcNotificationTypeSurpriseRemoval = 13,

	//! For internal use only.
	fcNotificationTypeStandby = 14,
} fcNotifyType, fcNotificationType;

//-----------------------------------------------------------------------------
//          Receive / Transmit
//-----------------------------------------------------------------------------

//! This enumeration contains the different packet types.
//! \see #fcPacket
typedef enum fcPacketType
{
    fcPacketTypeInfo           = 1,   //!< Frame is an info packet
    fcPacketTypeFlexRayFrame   = 2,   //!< Frame is a FlexRay frame
    fcPacketTypeError          = 3,   //!< Frame is an error packet
    fcPacketTypeStatus         = 4,   //!< Frame is a status packet
    fcPacketTypeTrigger        = 5,   //!< Frame is a trigger packet
    fcPacketTypeTxAcknowledge  = 6,   //!< Frame is a transmit acknowledge packet
    fcPacketTypeNMVector       = 7,   //!< Frame is a network management vector packet
    fcPacketTypeNotification   = 8,   //!< Frame is a notification packet.
    fcPacketTypeTriggerEx      = 9,   //!< Frame is a trigger packet (2. version)
	fcPacketTypeCAN			   = 10,  //!< Frame is a CAN packet
	fcPacketTypeCANError       = 11,  //!< Frame is a CAN error packet
} fcPacketType;


//-----------Info packet-------------------------------------------------------

//! This structure describes an information packet. This packet type informs you
//! about the start of a new cycle. All packets received between two
//! consecutive info packets are part of the current cycle.
//! \remarks A timestamp overflow occurs after approximately 4295 seconds.
typedef struct fcInfoPacket
{
    //!< The current cycle (FlexRay Protocol Specification:'vRF!Header!CycleCount')
	fcDword CurrentCycle;

	//!< The FlexCard time stamp (1 us resolution). Indicates the time at which
	//!< the packet was generated.
    fcDword TimeStamp;

	//!< Rate correction value (complement on two). Indicates by how many
	//!< microticks the node's cycle length should be changed.
	fcDword RateCorrection : 12;

	//!< Offset correction value (two' complement). Indicates the number of
	//!< microticks that are added to the offset correction segment of the
	//!< network idle time.
	fcDword OffsetCorrection : 19;

	//!< FlexRay Protocol Specification: vClockCorrectionFailed
	fcDword ClockCorrectionFailedCounter: 4;

	//!< FlexRay Protocol Specification: vAllowPassiveToActive
	fcDword	PassiveToActiveCount: 5;

    //!< The CC which created the packet.
	fcCC CC;
} fcInfoPacket;


//-----------FlexRay frame-----------------------------------------------------

//! This structure is equivalent to the FlexRay frame described in the FlexRay
//! protocol specification.
//! \note The payload length is a multiple of 16 bit words. The payload data is
//!		  also given in 16 bit words.
typedef struct fcFlexRayFrame
{
    //! The frame id defines the slot in which the frame was transmitted.
    //! (FlexRay Protocol Specification: 'vRF!Header!FrameID')
    fcDword ID : 11;

    //! Indicates if the frame is a start up frame (=1) or not (=0) (FlexRay
    //! Protocol Specification: 'vRF!Header!SuFIndicator')
    fcDword STARTUP : 1;

    //! Indicates if the frame is a sync frame (=1) or not (=0) (FlexRay
    //! Protocol Specification: 'vRF!Header!SyFIndicator')
    fcDword SYNC : 1;

    //! Set to 0, the null frame indicator indicates that pData contains no
    //! valid data. Set to 1, it indicates that pData contains valid data.
	//! (FlexRay Protocol Specification: 'vRF!Header!NFIndicator')
    fcDword NF : 1;

    //! The payload preamble indicator indicates whether or not an optional
	//! vector is contained within the payload segment of the frame transmitted.
	//! In the static segment, it indicates the presence of a network management
	//! vector at the beginning of the payload. In the dynamic segment it
	//! indicates the presence of a message id at the beginning of the payload,
	//! (FlexRay Protocol Specification: vRF!Header!PPIndicator).
    fcDword PP : 1;

    //! Reserved bit (FlexRay Protocol Specification: 'vRF!Header!Reserved')
    fcDword R : 1;

    //! Defines the number of 16 bit words contain in pData (FlexRay Protocol
	//! Specification: 'vRF!Header!Length')
    fcDword PayloadLength : 7;

    //! The cycle in which the frame was received. (FlexRay Protocol
    //! Specification: 'vRF!Header!CycleCount')
    fcDword CycleCount : 6;

    //! The header CRC contains the cyclic redundancy check code is computed
	//! over the sync frame indicator, the start up frame indicator, the frame
	//! id and the payload length.(FlexRay Protocol Specification:
	//! vRF!Header!HeaderCRC)
    fcDword HeaderCRC : 11;

    //! The pointer to the payload data. The payload is given in 16 bit words.
    //! (FlexRay Protocol Specification: 'vRF!Payload')
    fcWord* pData;

    //
    // Message Buffer slot status
    //

    //! The channel (A or B) on which the frame was received. (FlexRay Protocol
    //! Specification: 'vRF!Channel').
    fcChannel Channel;

    //! If a valid frame was received, this parameter is set to 1 (FlexRay
	//! Protocol Specification: vSS!ValidFrameA or vSS!ValidFrameB depends on
	//! Channel - Table 9-2: Slot status interpretation).
    fcDword ValidFrame : 1;

    //! If a syntax error was observed, this parameter is set to 1
    //! (frame is syntactically incorrect) (FlexRay Protocol Specification:
	//! 'vSS!SyntaxErrorA' or 'vSS!SyntaxErrorB' depends on Channel).
    fcDword SyntaxError : 1;

    //! If a content error was observed, this parameter is set to 1
    //! (frame is semantically incorrect) (FlexRay Protocol Specification:
	//! 'vSS!ContentErrorA' or 'vSS!ContentErrorB' depends on Channel).
    fcDword ContentError : 1;

    //! If a slot boundary violation was observed, this parameter is set to 1
	//! (FlexRay Protocol Specification: vSS!BViolationA or vSS!BViolationB
	//! depends on Channel).
    fcDword SlotBoundaryViolation : 1;

	//! If the packet was generated by the asynchronous debug mode, this
	//! parameter is set to 1.
	fcDword AsyncMode : 1;

	//! If the packet was generated by the asynchronous debug mode, the FrameCRC
	//! contains the cyclic redundancy check code is computed over complete frame.
	//! In synchronous monitoring mode, this parameter is not set.
	fcDword FrameCRC : 24;

	//!< The FlexCard time stamp (1 us resolution). The timestamp marks the
	//!< begin of the reception of the frame.
	fcDword TimeStamp;

    //!< The CC on which the frame was received.
	fcCC CC;
} fcFlexRayFrame;

//! This structure provides information about a transmit acknowledge packet.
//! Transmit acknowledge packets are used to inform the user when a frame is
//! transmitted.
typedef struct fcTxAcknowledgePacket
{
    //! The buffer id used to transmit the frame (equivalent to the buffer id
	//! returned by the function #fcbConfigureMessageBuffer).
    fcDword BufferId;

    //! FlexCard time stamp (1 us resolution). The timestamp marks the
	//!< begin of the transmission of the frame.
    fcDword TimeStamp;

    //! Indicates the cycle in which the frame was transmitted. (FlexRay
	//! Protocol Specification: vTF!Header!CycleCount)
    fcDword CycleCount;

	//! The frame id defines the slot in which the frame was transmitted.
    //! (FlexRay Protocol Specification: 'vRF!Header!FrameID')
    fcDword ID : 11;

    //! Indicates if the frame is a start up frame (=1) or not (=0) (FlexRay
    //! Protocol Specification: 'vRF!Header!SuFIndicator')
    fcDword STARTUP : 1;

    //! Indicates if the frame is a sync frame (=1) or not (=0) (FlexRay
    //! Protocol Specification: 'vRF!Header!SyFIndicator')
    fcDword SYNC : 1;

    //! Set to 0, the null frame indicator indicates that pData contains no
    //! valid data. Set to 1, it indicates that pData contains valid data.
	//! (FlexRay Protocol Specification: 'vRF!Header!NFIndicator')
    fcDword NF : 1;

    //! The payload preamble indicator indicates whether or not an optional
	//! vector is contained within the payload segment of the frame transmitted.
	//! In the static segment, it indicates the presence of a network management
	//! vector at the beginning of the payload. In the dynamic segment it
	//! indicates the presence of a message id at the beginning of the payload,
	//! (FlexRay Protocol Specification: vRF!Header!PPIndicator).
    fcDword PP : 1;

    //! Reserved bit (FlexRay Protocol Specification: 'vRF!Header!Reserved')
    fcDword R : 1;

    //! Defines the number of 16 bit words contain in pData (FlexRay Protocol
	//! Specification: 'vRF!Header!Length')
    fcDword PayloadLength : 7;

	//! If a valid frame was transmitted, this parameter is set to 1 (FlexRay
	//! Protocol Specification: vSS!ValidFrameA or vSS!ValidFrameB depends on
	//! Channel - Table 9-2: Slot status interpretation).
	fcDword ValidFrame : 1;

	//! If a syntax error was observed, this parameter is set to 1
	//! (frame is syntactically incorrect) (FlexRay Protocol Specification:
	//! 'vSS!SyntaxErrorA' or 'vSS!SyntaxErrorB' depends on Channel).
	fcDword SyntaxError : 1;

	//! If a content error was observed, this parameter is set to 1
	//! (frame is semantically incorrect) (FlexRay Protocol Specification:
	//! 'vSS!ContentErrorA' or 'vSS!ContentErrorB' depends on Channel).
	fcDword ContentError : 1;

    //! The header CRC contains the cyclic redundancy check code is computed
	//! over the sync frame indicator, the start up frame indicator, the frame
	//! id and the payload length.(FlexRay Protocol Specification:
	//! vRF!Header!HeaderCRC)
    fcDword HeaderCRC : 11;

    //! The pointer to the payload data. The payload is given in 16 bit words.
    //! (FlexRay Protocol Specification: 'vRF!Payload')
    fcWord* pData;

    //! The channel (A or B) on which the frame was transmitted. (FlexRay Protocol
    //! Specification: 'vRF!Channel')
    fcChannel Channel;

	//! The CC on which the frame was transmitted. (relevant for FlexCard PMC (II))
	fcCC CC;
} fcTxAcknowledgePacket;


//-----------Error packet------------------------------------------------------

//! This enumeration contains the different error types reported by an error
//! packet.
typedef enum fcErrorPacketFlag
{
    //! No error occurred
    fcErrNone = 0,

    //! FlexCard buffer overflow. This error occurs if the application was too
	//! slow to receive and process the packets. If the FlexCard is configured
	//! to stop the monitoring it is necessary to stop and start the monitoring
	//! again. Else the FlexCard continue the monitoring when an amount of free
	//! RAM space is available again. In such a case the FlexCard loses packets.
    fcErrFlexcardOverflow,

    //! Protocol Operation Control error. Additional information are described in
    //! the structure fcErrPOCErrorModeChangedInfo
    fcErrPOCErrorModeChanged,

    //! Additional information are described in the structure fcErrSyncFramesInfo
    fcErrSyncFramesBelowMinimum,

    //! Additional information are described in the structure fcErrSyncFramesInfo
    fcErrSyncFrameOverflow,

    //! Additional information are described in the structure fcErrClockCorrectionFailureInfo
    fcErrClockCorrectionFailure,

    //! Internal E-Ray error. No additional information is available
    fcErrParityError,

    //! The following four errors are internal FlexCard errors. For these errors
    //! no additional information exists (fcErrorPacket.AdditionalInfo is not
    //! valid)
    fcErrReceiveFIFOOverrun,
    fcErrEmptyFIFOAccess,
    fcErrIllegalInputBufferAccess,
    fcErrIllegalOutputbufferAccess,

    //! A syntax error was observed in the configured slot (frame is  syntactically
    //! incorrect). (FlexRay Protocol Specification: 'vSS!SyntaxErrorA' or
    //! 'vSS!SyntaxErrorB' depends on Channel) Additional information described
    //! in the structure fcErrSlotInfo
    fcErrSyntax,

    //! A content error was observed in the configured slot (frame is
    //! semantically incorrect). (FlexRay Protocol Specification:
	//! 'vSS!ContentErrorA' or 'vSS!ContentErrorB' depends on Channel)
    //! Additional information described in the structure fcErrSlotInfo
    fcErrContent,

    //! A slot boundary violation was observed. (FlexRay Protocol Specification:
    //! 'vSS!BViolationA' or 'vSS!BViolationB' depends on Channel)
    //! Additional information described in the structure fcErrSlotInfo
    fcErrSlotBoundaryViolation,

    //! Additional information are described in the structure fcErrSlotInfo
    fcErrTransmissionAcrossBoundary,

    //! Additional information are described in the structure fcErrSlotInfo
    fcErrLatestTransmitViolation,

	//! Syntax error in symbol window was observed. Additional information are
	//! described  in the structure fcErrSlotInfo.
	fcErrSyntaxSW,

	//! Slot boundary violation in symbol window was observed. Additional
	//! information are described  in the structure fcErrSlotInfo.
	fcErrSlotBoundaryViolationSW,

	//! Transmission conflict in symbol window was observed. Additional
	//! information are described  in the structure fcErrSlotInfo.
	fcErrTransmissionConflictSW,

	//! Syntax error in network idle time was observed. Additional information
	//! are described  in the structure fcErrSlotInfo.
	fcErrSyntaxNIT,

	//! Slot boundary violation in network idle time was observed. Additional
	//! information are described  in the structure fcErrSlotInfo.
	fcErrSlotBoundaryViolationNIT,
} fcErrorPacketFlag;

 //! This structure provides additional information about the
 //! fcErrPOCErrorModeChanged error.
typedef struct fcErrPOCErrorModeChangedInfo
{
    //! Contains the new POC error mode (HALT, NORMAL_ACTIVE or NORMAL_PASSIVE)
	fcState State;
} fcErrPOCErrorModeChangedInfo;


//! This structure provides additional information about the
//! fcErrSyncFramesBelowMinimum and fcErrSyncFrameOverflow errors.
typedef struct fcErrSyncFramesInfo
{
    //! Valid sync frame received and transmitted on channel A in even
	//! communication cycles
	fcDword SyncFramesEvenA : 4;
    //! Valid sync frame received and transmitted on channel B in even
	//! communication cycles
	fcDword SyncFramesEvenB : 4;
    //! Valid sync frame received and transmitted on channel A in odd
	//! communication cycles
	fcDword SyncFramesOddA : 4;
	//! Valid sync frame received and transmitted on channel B in odd
	//! communication cycles
    fcDword SyncFramesOddB : 4;
} fcErrSyncFramesInfo;

//! This structure provides additional information about the
//! fcErrClockCorrectionFailure error.
typedef struct fcErrClockCorrectionFailureInfo
{
    //! Is set to 1 if no rate correction can be performed because no pairs of
	//! even/odd sync frames were received.
	fcDword MissingRateCorrection : 1;
   	//! Is set to 1 if the maximum rate correction limit is reached.
	fcDword RateCorrectionLimitReached : 1;
    //! Is set to 1 if the maximum offset correction limit is reached.
	fcDword OffsetCorrectionLimitReached : 1;
	//! Is set to 1 if no offset correction can be performed because no sync
	//! frames were received.
	fcDword MissingOffsetCorrection: 1;
    //! Valid sync frame received and transmitted on channel A in even
	//! communication cycles
	fcDword SyncFramesEvenA : 4;
    //! Valid sync frame received and transmitted on channel B in even
	//! communication cycles
	fcDword SyncFramesEvenB : 4;
    //! Valid sync frame received and transmitted on channel A in odd
	//! communication cycles
	fcDword SyncFramesOddA : 4;
	//! Valid sync frame received and transmitted on channel B in odd
	//! communication cycles
    fcDword SyncFramesOddB : 4;
} fcErrClockCorrectionFailureInfo;

//! This structure provides additional information about the fcErrSyntax,
//! fcErrContent, fcErrSlotBoundaryViolation, fcErrTransmissionAcrossBoundary,
//! fcErrLatestTransmitViolation, fcErrSyntaxSW, fcErrSlotBoundaryViolationSW,
//! fcErrTransmissionConflictSW, fcErrSyntaxNIT and fcErrSlotBoundaryViolationNIT
//! errors.
typedef struct fcErrSlotInfo
{
	//! The channel on which the error was observed.
	fcChannel Channel;

	//! The approximate slot count when the error occurred.
	fcDword SlotCount;
} fcErrSlotInfo;

//! This structure provides information about an error packet.
typedef struct fcErrorPacket
{
    fcErrorPacketFlag Flag;  //!< Error type
    fcDword TimeStamp;       //!< The FlexCard time stamp (1 us resolution).
							 //!< Indicates the time at which the packet was
							 //!< generated.
    fcDword CycleCount;      //!< The cycle in which the error occurred.

    union AdditionalInfo
    {
		//! Additional information about the fcErrPOCErrorModeChanged error
		fcErrPOCErrorModeChangedInfo ErrPOCErrorModeChangedInfo;

		//! Additional information about the fcErrSyncFramesBelowMinimum ,
		//! fcErrSyncFrameOverflow errors
		fcErrSyncFramesInfo ErrSyncFramesInfo;

		//! Additional information about the fcErrSyntax, fcErrContent,
		//! fcErrSlotBoundaryViolation, fcErrTransmissionAcrossBoundary,
		//! fcErrLatestTransmitViolation, fcErrSyntaxSW,
		//! fcErrSlotBoundaryViolationSW, fcErrTransmissionConflictSW,
		//! fcErrSyntaxNIT and fcErrSlotBoundaryViolationNIT errors.
		fcErrSlotInfo ErrSlotInfo;

		//! Additional information about the fcErrClockCorrectionFailure error
		fcErrClockCorrectionFailureInfo ErrClockCorrectionFailureInfo;
    } AdditionalInfo;

	//! The CC on which the packet was created.
	fcCC CC;

	fcDword Reserved;
} fcErrorPacket;

//-----------Status packet-----------------------------------------------------

//! Possible status flags from the hardware which are reported by a status
//! packet. Status packets are not supported by the Freescale FlexCard version.
typedef enum fcStatusPacketFlag
{
    fcStatusNone = 0,					 //!< No status change.
	fcStatusWakeupStatus,				 //!< Wakeup status has changed.
    fcStatusCollisionAvoidanceSymbol,    //!< Collision avoidance symbol was
										 //!< received
	fcStatusStartupCompletedSuccessfully,//!< Start up has been succesfully completed.
    fcStatusWakeupPatternChannelA,		 //!< Wakeup pattern received on Channel A
    fcStatusWakeupPatternChannelB,		 //!< Wakeup pattern received on Channel B
    fcStatusMTSReceivedonChannelA,       //!< Media Access Test Symbol received
										 //!< on Channel A
    fcStatusMTSReceivedonChannelB,       //!< Media Access Test Symbol received
										 //!< on Channel B
} fcStatusPacketFlag;

//! This structure provides additional information about the cStatusWakeupStatus
//! status.
typedef struct fcStatusWakeupInfo
{
    //! Current wakeup state
	fcWakeupStatus WakeupStatus;
} fcStatusWakeupInfo;

//! This structure provides information about a status packet.
typedef struct fcStatusPacket
{
    fcStatusPacketFlag Flag;  //!< Status type
    fcDword TimeStamp;        //!< The FlexCard time stamp (1 us resolution).
							  //!< Indicates the time at which the packet was
							  //!< generated.
    fcDword CycleCount;       //!< The cycle in which the status has changed.

    union
    {
        //! Additional information about fcStatusWakeupStatus status
		fcStatusWakeupInfo StatusWakeupInfo;
    } AdditionalInfo;

	//! The CC on which the packet was created.
	fcCC CC;

    fcDword Reserved[2];
} fcStatusPacket;


//-----------Trigger packet----------------------------------------------------

//! This structure provides information about a trigger packet ex.
typedef struct fcTriggerExInfoPacket
{
    //! The fulfilled condition which has caused the trigger packet generation.
	fcDword Condition;
    //! The FlexCard time stamp (1 us resolution). Indicates the time at which
	//! the packet was generated.
	fcDword TimeStamp;

	//! Sequence count for each signal
	fcDword SequenceCount;

	//! Reserved for future use
	fcDword Reserved1;
	//! Variable that receives the current performance-counter value. This
	//! value is only valid for software triggers and FlexCard Windows driver.
	fcQuad PerformanceCounter;

	//! The edge on which the trigger was signaled.
	//! This value is only valid for hardware triggers.
	fcDword Edge;

	//! The trigger line which detected a trigger signal
	//! This value is only valid for hardware triggers of FlexCard PMC (II) only.
	fcDword TriggerLine;

	//! Reserved for future use
	fcDword Reserved[4];
} fcTriggerExInfoPacket;

//! THIS STRUCTURE IS OBSOLETE  (Please use #fcTriggerExInfoPacket now.)
//! This structure provides information about a trigger packet.
typedef struct fcTriggerInfoPacket
{
    //! Type of trigger info packet
    fcTriggerType Type;

    //! The fulfilled condition which has caused the trigger packet generation.
    fcTriggerCondition Condition;

    //! The FlexCard time stamp (1 us resolution). Indicates the time at which
	//! the packet was generated.
    fcDword TimeStamp;

    //! Sequence count for each signal
    fcDword SequenceCount;

    //! Variable that receives the current performance-counter value. This value
	//! is only valid for software triggers (fcTriggerTypeSoftware).
    fcQuad PerformanceCounter;

    fcDword Reserved;
} fcTriggerInfoPacket;

//-----------Network Management Vector packet----------------------------------

//! This structure provides information about a network management vector.
//! (FlexRay Protocol Specification: NMVector)
typedef struct fcNMVectorPacket
{
	//! The FlexCard time stamp (1 us resolution). Indicates the time at which
	//! the packet was generated.
	fcDword TimeStamp;

	//! The cycle in which the network management vector was updated.
	fcDword CycleCount;

    //! Length of network management vector in number of bytes. (FlexRay
    //! Protocol Specification: 'gNetworkManagementVectorLength')
    fcDword NMVectorLength;

    //! The data bytes of the network management vector. The number of bytes
    //! which are valid in this vector depend on the parameter
    //! NMVectorLength.
    fcByte NMVector[12];

	//! The CC on which the packet was created.
	fcCC CC;

	//! Reserved for future use
    fcDword Reserved;
} fcNMVectorPacket;

//-----------Notification packet----------------------------------------------------

//! This structure provides information about a notification packet, provides
//! information about the time when the timer elapses. A notification packet is
//! generated each time the configured time out elapses. The generation of this
//! packet can be controlled with the function #fcbNotificationPacket.
typedef struct fcNotificationPacket
{
    //! The FlexCard time stamp (1 us resolution). Indicates the time at which
	//! the packet was generated.
    fcDword TimeStamp;

    //! This parameter is incremented each time a notification packet is
	//! generated.
    fcDword SequenceCounter;

	//! Reserved for future use
    fcDword Reserved;
} fcNotificationPacket;

//----------- CAN packet----------------------------------------------------

//! This structure provides information about a CAN packet.
typedef struct fcCANPacket
{
	//! The CAN message identifier which was received or transmitted.
	fcDword ID           :29;
	//! If this flag is 1 the CAN message is an extended frame. If set to 0 it
	//! is a standard frame.
	fcDword ExtendedId   :1;
	//! The FlexCard time stamp (1 us resolution). Indicates the time at which
	//! the packet was generated.
	fcDword TimeStamp;
	//! Indicates the corresponding buffer number for the CAN packet.
	fcDword BufferNumber :8;
	//! Indicates the data length (in bytes).
	fcDword DLC          :4;
	//! This flag depends on the parameter RemoteFrame. If Direction is 0 and
	//! RemoteFrame is 0, the CAN packet is a received data frame. If Direction
	//! is 1 and RemoteFrame is 0 the CAN packet is a transmit acknowledge frame
	//! generated by the FlexCard.
	fcDword Direction    :1;
	//! This flag depends on the parameter Direction. If RemoteFrame is 1 and
	//! Direction is 0, the CAN packet is a remote rx frame. If RemoteFrame is
	//! 1 and Direction is 1, the CAN packet is a remote tx frame.
	fcDword RemoteFrame  :1;
	//! If this flag is 1 the CAN communication controller has lost a message.
	//! If 0 no message has lost. This flag is only valid with Direction = 0.
	fcDword MessageLost  :1;
	//! Reserved for future use.
	fcDword Reserved;
	//! The CAN communication controller on which the frame was received or transmitted.
	fcCC CC;
	//! The received or transmitted data. All of the 8 data bytes can be read. The
	//! corresponding parameter DLC indicates the length of the valid values.
	fcByte Data[8];
} fcCANPacket;


//----------- CAN error packet----------------------------------------------------

//! This enumeration contains the different error types reported by a CAN error packet.
typedef enum fcCANErrorType
{
	//! No error occurred.
	fcCANErrorNone = 0,

	//! CAN bus errors.

	//! More than 5 equal bits in a sequence have occurred in a part of a received
	//! message where this is not allowed.
	fcCANErrorStuff,

	//! A fixed format part of a received frame has the wrong format.
	fcCANErrorForm,

	//! The message the CAN communication controller transmitted was not acknowledged
	//! by another node.
	fcCANErrorAcknowledge,

	//! During the transmission of a message (with the exception of the arbitration
	//! field), the device wanted to send a recessive level (bit of logical value 1),
	//! but the monitored bus value was dominant (bit of logical value 0).
	fcCANErrorBit1,

	//! During the transmission of a message, the device wanted to send a dominant level
	//! (data or identifier bit logical value 0), but the monitored bus value was
	//! recessive (data or identifier bit logical value 1).
	fcCANErrorBit0,

	//! The CRC check sum was incorrect in the message received, the CRC received for
	//! an incoming message does not match with the calculated CRC for the received data.
	fcCANErrorCRC,

	//! The Parity Check Mechanism has detected a parity error in the Message RAM.
	fcCANErrorParity,
} fcCANErrorType;

//! This structure provides information about a CAN error packet.
typedef struct fcCANErrorPacket
{
	fcCANErrorType Type;			//!< Error type
	fcCANCcState State;				//!< Current CAN CC state
	fcDword TimeStamp;				//!< The FlexCard time stamp (1 us resolution).
									//!< Indicates the time at which the packet was
									//!< generated.
	fcDword ReceiveErrorCounter;	//!< Actual state of the Receive Error Counter.
									//!> Valid values range from 0 to 127.
	fcDword TransmitErrorCounter;	//!< Actual state of the Transmit Error Counter.
									//!< Values values range 0 to 255.
	fcCC CC;						//!< The CC on which the packet was created.
	fcDword Reserved[2];			//!< Reserved for future use.
} fcCANErrorPacket;

//-----------------------------------------------------------------------------

//! This structure provides information about a packet.
typedef struct fcPacket
{
    //! Type of packet.
    fcPacketType Type;

    //! Pointer to the packet data. The content depends on the packet type.
    union
    {
        fcFlexRayFrame*         FlexRayFrame;
        fcInfoPacket*           InfoPacket;
        fcErrorPacket*          ErrorPacket;
        fcStatusPacket*         StatusPacket;
        fcTriggerInfoPacket*    TriggerPacket;
        fcTxAcknowledgePacket*  TxAcknowledgePacket;
        fcNMVectorPacket*       NMVectorPacket;
        fcNotificationPacket*   NotificationPacket;
		fcTriggerExInfoPacket*  TriggerExPacket;
		fcCANPacket*            CANPacket;
		fcCANErrorPacket*       CANErrorPacket;
    };

    //! Pointer to the next packet. If the pointer is NULL, there are no more
    //! packets available.
    struct fcPacket*  pNextPacket;
} fcPacket;

#endif /* #define _FCBASE_TYPES_H_ */
